<?php
/**
 * Admin Panel Handler
 * Version: 2.0
 */

declare(strict_types=1);

class AdminPanel {
    private Bot $bot;
    private Telegram $telegram;
    private Database $db;
    
    public function __construct(Bot $bot) {
        $this->bot = $bot;
        $this->telegram = $bot->getTelegram();
        $this->db = $bot->getDb();
    }
    
    public function handle(): void {
        $text = $this->bot->getText();
        
        if (!$text) {
            return;
        }
        
        $command = $this->parseCommand($text);
        
        switch ($command) {
            case 'start':
                $this->showMainPanel();
                break;
                
            case 'panel':
            case 'پنل':
                $this->showMainPanel();
                break;
                
            case 'stats':
            case 'آمار':
                $this->showStats();
                break;
                
            case 'broadcast':
            case 'ارسال همگانی':
                $this->handleBroadcast();
                break;
                
            case 'groups':
            case 'گروه‌ها':
                $this->showGroups();
                break;
                
            case 'users':
            case 'کاربران':
            case 'کاربران':
                $this->showUsers();
                break;
                
            case 'subscriptions':
            case 'اشتراک‌ها':
                $this->showSubscriptions();
                break;
                
            case 'receipts':
            case 'رسیدها':
                $this->showPendingReceipts();
                break;
                
            case 'plans':
            case 'پلن‌ها':
                $this->showPlansManagement();
                break;
                
            case 'payment_settings':
            case 'تنظیمات_پرداخت':
            case 'تنظیمات پرداخت':
                $this->showPaymentSettings();
                break;
                
            case 'bot_mode':
            case 'حالت_ربات':
            case 'حالت ربات':
                $this->showBotMode();
                break;
                
            case 'toggle_subscription':
                $this->toggleSubscriptionMode();
                break;
                
            case 'settings':
            case 'تنظیمات':
                $this->showMainPanel(); // فعلاً به پنل اصلی برمیگرده
                break;
                
            case 'logs':
            case 'لاگ‌ها':
            case 'لاگها':
            case 'لاگ ها':
                $this->showLogs();
                break;
                
            case 'back':
            case 'بازگشت':
                $this->showMainPanel();
                break;
        }
    }
    
    private function showMainPanel(): void {
        $keyboard = [
            'keyboard' => [
                [['text' => '📊 آمار'], ['text' => '👥 کاربران']],
                [['text' => '🏘 گروه‌ها'], ['text' => '📢 ارسال همگانی']],
                [['text' => '💎 اشتراک‌ها'], ['text' => '🧾 رسیدها']],
                [['text' => '📦 پلن‌ها'], ['text' => '💳 تنظیمات پرداخت']],
                [['text' => '🔄 حالت ربات'], ['text' => '⚙️ تنظیمات']],
                [['text' => '📝 لاگ‌ها'], ['text' => '🔙 بازگشت']]
            ],
            'resize_keyboard' => true
        ];
        
        $mode = SUBSCRIPTION_REQUIRED ? '💎 اشتراکی' : '🆓 رایگان';
        
        $message = "🎛 *پنل مدیریت*\n\n" .
                   "به پنل مدیریت ربات خوش آمدید.\n" .
                   "حالت فعلی: {$mode}\n\n" .
                   "بخش مورد نظر را انتخاب کنید:";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showStats(): void {
        $totalGroups = $this->db->query(
            "SELECT COUNT(*) as count FROM `groups`"
        )->fetch_assoc()['count'];
        
        $totalMembers = $this->db->query(
            "SELECT COUNT(*) as count FROM `members`"
        )->fetch_assoc()['count'];
        
        $todayGroups = $this->db->query(
            "SELECT COUNT(*) as count FROM `groups` WHERE DATE(created_at) = CURDATE()"
        )->fetch_assoc()['count'];
        
        $todayMembers = $this->db->query(
            "SELECT COUNT(*) as count FROM `members` WHERE DATE(joined_at) = CURDATE()"
        )->fetch_assoc()['count'];
        
        $message = "📊 *آمار کامل ربات*\n\n" .
                   "*کل:*\n" .
                   "👥 کاربران: `{$totalMembers}`\n" .
                   "🏘 گروه‌ها: `{$totalGroups}`\n\n" .
                   "*امروز:*\n" .
                   "👤 کاربران جدید: `{$todayMembers}`\n" .
                   "🏘 گروه‌های جدید: `{$todayGroups}`\n\n" .
                   "📅 تاریخ: " . date('Y-m-d H:i:s');
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    private function handleBroadcast(): void {
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '👥 ارسال به کاربران', 'callback_data' => 'broadcast_users'],
                    ['text' => '🏘 ارسال به گروه‌ها', 'callback_data' => 'broadcast_groups']
                ],
                [
                    ['text' => '❌ انصراف', 'callback_data' => 'cancel']
                ]
            ]
        ];
        
        $message = "📢 *ارسال همگانی*\n\n" .
                   "پیام خود را فوروارد کنید یا بنویسید.\n" .
                   "سپس مقصد را انتخاب کنید:";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showGroups(): void {
        $groups = $this->db->fetchAll(
            "SELECT id, created_at FROM `groups` ORDER BY created_at DESC LIMIT 10"
        );
        
        $message = "🏘 *لیست گروه‌ها (10 تای آخر)*\n\n";
        
        foreach ($groups as $group) {
            $message .= "• `{$group['id']}` - {$group['created_at']}\n";
        }
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    private function showUsers(): void {
        $users = $this->db->fetchAll(
            "SELECT id, username, joined_at FROM `members` ORDER BY joined_at DESC LIMIT 10"
        );
        
        $message = "👥 *لیست کاربران (10 نفر آخر)*\n\n";
        
        foreach ($users as $user) {
            $username = $user['username'] ? "@{$user['username']}" : "بدون یوزرنیم";
            $message .= "• `{$user['id']}` - {$username}\n";
        }
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    public function handleCallback(): void {
        $data = $this->bot->getCallbackData();
        
        // Receipt approval/rejection
        if (preg_match('/^approve_receipt_(\d+)$/', $data, $matches)) {
            $this->approveReceipt((int)$matches[1]);
            return;
        } elseif (preg_match('/^reject_receipt_(\d+)$/', $data, $matches)) {
            $this->rejectReceipt((int)$matches[1]);
            return;
        } elseif (preg_match('/^view_receipt_(\d+)$/', $data, $matches)) {
            $this->viewReceipt((int)$matches[1]);
            return;
        }
        
        switch ($data) {
            case 'locks':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id, "قفل‌ها - در حال توسعه");
                break;
                
            case 'lists':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id, "لیست‌ها - در حال توسعه");
                break;
                
            case 'settings':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
                $this->telegram->editMessageText(
                    $this->bot->getChatId(),
                    $this->bot->getMessageId(),
                    "⚙️ *تنظیمات*\n\nاز دکمه‌های زیر استفاده کنید:",
                    'Markdown'
                );
                break;
                
            case 'stats':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
                $this->showStats();
                break;
                
            case 'toggle_mode':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id, "برای تغییر حالت، فایل config.php را ویرایش کنید", true);
                break;
                
            case 'advanced_mode_settings':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id, "تنظیمات پیشرفته - در حال توسعه");
                break;
                
            case 'broadcast_users':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
                $this->startBroadcastUsers();
                break;
                
            case 'broadcast_groups':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
                $this->startBroadcastGroups();
                break;
                
            case 'cancel':
            case 'close':
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
                $this->telegram->deleteMessage(
                    $this->bot->getChatId(),
                    $this->bot->getMessageId()
                );
                break;
                
            default:
                $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id, "دکمه شناخته نشد: {$data}");
                break;
        }
    }
    
    private function startBroadcastUsers(): void {
        // Save broadcast task to database
        $this->telegram->answerCallbackQuery(
            $this->bot->getCallbackQuery()->id,
            "✅ ارسال به کاربران شروع شد!"
        );
    }
    
    private function startBroadcastGroups(): void {
        // Save broadcast task to database
        $this->telegram->answerCallbackQuery(
            $this->bot->getCallbackQuery()->id,
            "✅ ارسال به گروه‌ها شروع شد!"
        );
    }
    
    private function parseCommand(string $text): string {
        $text = trim($text);
        // Remove emojis and special characters
        $text = preg_replace('/[\x{1F300}-\x{1F9FF}]/u', '', $text);
        $text = preg_replace('/^[\/\!\?#@]+/', '', $text);
        $text = trim($text);
        return strtolower($text);
    }
    
    /**
     * Show subscriptions overview
     */
    private function showSubscriptions(): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $stats = $subManager->getStats();
        
        $message = "💎 *مدیریت اشتراک‌ها*\n\n" .
                   "✅ اشتراک‌های فعال: `{$stats['active']}`\n" .
                   "⏳ در انتظار تأیید: `{$stats['pending']}`\n" .
                   "⚠️ منقضی شونده (7 روز): `{$stats['expiring_soon']}`\n" .
                   "💰 درآمد کل: `" . number_format($stats['revenue']) . "` تومان";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🧾 رسیدهای در انتظار', 'callback_data' => 'pending_receipts']],
                [['text' => '📦 مدیریت پلن‌ها', 'callback_data' => 'manage_plans']],
                [['text' => '💳 تنظیمات پرداخت', 'callback_data' => 'payment_settings']]
            ]
        ];
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    /**
     * Show pending receipts
     */
    private function showPendingReceipts(): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $receipts = $subManager->getPendingReceipts(5);
        
        if (empty($receipts)) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                "✅ رسید در انتظار تأییدی وجود ندارد!"
            );
            return;
        }
        
        foreach ($receipts as $receipt) {
            $userName = $receipt['first_name'] ?? 'کاربر';
            $username = $receipt['username'] ? "@{$receipt['username']}" : '';
            
            $message = "🧾 *رسید #{$receipt['id']}*\n\n" .
                       "👤 کاربر: {$userName} {$username}\n" .
                       "🆔 آیدی: `{$receipt['user_id']}`\n" .
                       "📦 پلن: {$receipt['plan_name']}\n" .
                       "💰 مبلغ: " . number_format($receipt['price']) . " تومان\n" .
                       "📅 تاریخ: " . date('Y/m/d H:i', strtotime($receipt['submitted_at']));
            
            if ($receipt['transaction_id']) {
                $message .= "\n🔢 شماره تراکنش: `{$receipt['transaction_id']}`";
            }
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '✅ تأیید', 'callback_data' => "approve_receipt_{$receipt['id']}"],
                        ['text' => '❌ رد', 'callback_data' => "reject_receipt_{$receipt['id']}"]
                    ],
                    [['text' => '👁 مشاهده رسید', 'callback_data' => "view_receipt_{$receipt['id']}"]]
                ]
            ];
            
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                $message,
                'Markdown',
                $keyboard
            );
        }
    }
    
    /**
     * View receipt image
     */
    private function viewReceipt(int $receiptId): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $receipt = $subManager->getReceipt($receiptId);
        
        if (!$receipt || !$receipt['receipt_image']) {
            $this->telegram->answerCallbackQuery(
                $this->bot->getCallbackQuery()->id,
                "❌ رسید یافت نشد!",
                true
            );
            return;
        }
        
        $caption = "🧾 رسید #{$receiptId}\n" .
                   "👤 کاربر: {$receipt['user_id']}\n" .
                   "📦 پلن: {$receipt['plan_name']}\n" .
                   "💰 مبلغ: " . number_format($receipt['price']) . " تومان";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '✅ تأیید', 'callback_data' => "approve_receipt_{$receiptId}"],
                    ['text' => '❌ رد', 'callback_data' => "reject_receipt_{$receiptId}"]
                ]
            ]
        ];
        
        $this->telegram->sendPhoto(
            $this->bot->getChatId(),
            $receipt['receipt_image'],
            $caption,
            'Markdown',
            $keyboard
        );
    }
    
    /**
     * Approve receipt
     */
    private function approveReceipt(int $receiptId): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $success = $subManager->approveReceipt($receiptId, $this->bot->getUserId());
        
        if ($success) {
            $receipt = $subManager->getReceipt($receiptId);
            
            $this->telegram->answerCallbackQuery(
                $this->bot->getCallbackQuery()->id,
                "✅ رسید تأیید شد!",
                false
            );
            
            // Notify user
            $this->telegram->sendMessage(
                $receipt['user_id'],
                "✅ *رسید شما تأیید شد!*\n\n" .
                "🎉 اشتراک شما فعال شد.\n" .
                "📦 پلن: {$receipt['plan_name']}\n" .
                "⏱ مدت: {$receipt['duration_days']} روز\n\n" .
                "از خرید شما متشکریم! 🙏"
            );
            
            // Update message
            $this->telegram->editMessageText(
                $this->bot->getChatId(),
                $this->bot->getMessageId(),
                "✅ رسید #{$receiptId} تأیید شد."
            );
        } else {
            $this->telegram->answerCallbackQuery(
                $this->bot->getCallbackQuery()->id,
                "❌ خطا در تأیید رسید!",
                true
            );
        }
    }
    
    /**
     * Reject receipt
     */
    private function rejectReceipt(int $receiptId): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $success = $subManager->rejectReceipt(
            $receiptId,
            $this->bot->getUserId(),
            "رسید توسط ادمین رد شد"
        );
        
        if ($success) {
            $receipt = $subManager->getReceipt($receiptId);
            
            $this->telegram->answerCallbackQuery(
                $this->bot->getCallbackQuery()->id,
                "❌ رسید رد شد!",
                false
            );
            
            // Notify user
            $this->telegram->sendMessage(
                $receipt['user_id'],
                "❌ *رسید شما رد شد*\n\n" .
                "متأسفانه رسید ارسالی شما تأیید نشد.\n\n" .
                "لطفاً:\n" .
                "• مبلغ صحیح را واریز کنید\n" .
                "• رسید واضح و خوانا ارسال کنید\n" .
                "• در صورت سوال با پشتیبانی تماس بگیرید"
            );
            
            // Update message
            $this->telegram->editMessageText(
                $this->bot->getChatId(),
                $this->bot->getMessageId(),
                "❌ رسید #{$receiptId} رد شد."
            );
        } else {
            $this->telegram->answerCallbackQuery(
                $this->bot->getCallbackQuery()->id,
                "❌ خطا در رد رسید!",
                true
            );
        }
    }
    
    /**
     * Show plans management
     */
    private function showPlansManagement(): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $plans = $subManager->getAllPlans();
        
        $message = "📦 *مدیریت پلن‌ها*\n\n";
        
        if (empty($plans)) {
            $message .= "هیچ پلنی وجود ندارد.";
        } else {
            foreach ($plans as $plan) {
                $message .= "💎 {$plan['name']}\n" .
                           "⏱ {$plan['duration_days']} روز\n" .
                           "💰 " . number_format($plan['price']) . " تومان\n\n";
            }
        }
        
        $message .= "\n📝 برای افزودن پلن جدید:\n" .
                    "`/addplan نام|نام_انگلیسی|مدت|قیمت`\n\n" .
                    "مثال:\n" .
                    "`/addplan یک ماهه|Monthly|30|50000`";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    /**
     * Show payment settings
     */
    private function showPaymentSettings(): void {
        require_once 'SubscriptionManager.php';
        $subManager = SubscriptionManager::getInstance();
        
        $settings = $subManager->getPaymentSettings();
        
        if ($settings) {
            $cardNumber = chunk_split($settings['card_number'], 4, '-');
            $cardNumber = rtrim($cardNumber, '-');
            
            $message = "💳 *تنظیمات پرداخت فعلی*\n\n" .
                       "شماره کارت: `{$cardNumber}`\n" .
                       "به نام: {$settings['card_holder']}\n";
            
            if ($settings['bank_name']) {
                $message .= "بانک: {$settings['bank_name']}\n";
            }
        } else {
            $message = "⚠️ تنظیمات پرداخت وجود ندارد.";
        }
        
        $message .= "\n\n📝 برای تنظیم/تغییر:\n" .
                    "`/setpayment شماره_کارت|نام_صاحب_کارت|نام_بانک`\n\n" .
                    "مثال:\n" .
                    "`/setpayment 6037991234567890|علی احمدی|ملی`";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    /**
     * Show bot mode settings
     */
    private function showBotMode(): void {
        $mode = SUBSCRIPTION_REQUIRED ? 'اشتراکی 💎' : 'رایگان 🆓';
        $trialDays = FREE_TRIAL_DAYS;
        
        $message = "🔄 *حالت ربات*\n\n" .
                   "حالت فعلی: *{$mode}*\n";
        
        if (SUBSCRIPTION_REQUIRED) {
            $message .= "🎁 دوره آزمایشی: {$trialDays} روز\n\n";
            $message .= "*قابلیت‌های نیازمند اشتراک:*\n";
            
            foreach (SUBSCRIPTION_FEATURES as $feature => $required) {
                if ($required) {
                    $featureNames = [
                        'ai_chat' => 'چت با AI',
                        'ai_advanced' => 'AI پیشرفته',
                        'plugins' => 'پلاگین‌ها',
                        'broadcast' => 'ارسال همگانی'
                    ];
                    $name = $featureNames[$feature] ?? $feature;
                    $message .= "✅ {$name}\n";
                }
            }
        } else {
            $message .= "\n✅ تمام قابلیت‌ها رایگان هستند";
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => SUBSCRIPTION_REQUIRED ? '🆓 تبدیل به رایگان' : '💎 تبدیل به اشتراکی', 'callback_data' => 'toggle_mode']],
                [['text' => '⚙️ تنظیمات پیشرفته', 'callback_data' => 'advanced_mode_settings']]
            ]
        ];
        
        $message .= "\n\n⚠️ *توجه:* تغییر حالت نیاز به ویرایش فایل config.php دارد.";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    /**
     * Toggle subscription mode (guide)
     */
    private function toggleSubscriptionMode(): void {
        $currentMode = SUBSCRIPTION_REQUIRED ? 'اشتراکی' : 'رایگان';
        $newMode = SUBSCRIPTION_REQUIRED ? 'رایگان' : 'اشتراکی';
        
        $message = "🔄 *تغییر حالت ربات*\n\n" .
                   "حالت فعلی: *{$currentMode}*\n" .
                   "حالت جدید: *{$newMode}*\n\n" .
                   "📝 *مراحل تغییر:*\n\n" .
                   "1️⃣ فایل `config.php` را باز کنید\n\n" .
                   "2️⃣ خط زیر را پیدا کنید:\n" .
                   "`define('SUBSCRIPTION_REQUIRED', " . (SUBSCRIPTION_REQUIRED ? 'true' : 'false') . ");`\n\n" .
                   "3️⃣ آن را به این تغییر دهید:\n" .
                   "`define('SUBSCRIPTION_REQUIRED', " . (SUBSCRIPTION_REQUIRED ? 'false' : 'true') . ");`\n\n" .
                   "4️⃣ فایل را ذخیره کنید\n\n" .
                   "✅ تغییرات بلافاصله اعمال می‌شود!";
        
        if (!SUBSCRIPTION_REQUIRED) {
            $message .= "\n\n💡 *نکته:* قبل از فعال کردن حالت اشتراکی:\n" .
                       "• حداقل یک پلن تعریف کنید\n" .
                       "• شماره کارت را تنظیم کنید\n" .
                       "• دوره آزمایشی را تنظیم کنید";
        }
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    /**
     * Show logs
     */
    private function showLogs(): void {
        $logFiles = glob(LOGS_PATH . '/*.log');
        
        if (empty($logFiles)) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                "⚠️ هیچ لاگی یافت نشد!"
            );
            return;
        }
        
        $message = "📋 *لاگ‌های موجود:*\n\n";
        
        foreach ($logFiles as $logFile) {
            $fileName = basename($logFile);
            $fileSize = filesize($logFile);
            $fileTime = date('Y/m/d H:i', filemtime($logFile));
            
            $message .= "📄 `{$fileName}`\n";
            $message .= "   حجم: " . number_format($fileSize) . " بایت\n";
            $message .= "   تاریخ: {$fileTime}\n\n";
        }
        
        $message .= "برای مشاهده لاگ‌ها:\n";
        $message .= "https://" . $_SERVER['HTTP_HOST'] . "/bot-v2/view_logs.php";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
}
