<?php
/**
 * توابع مربوط به دستورات کاربری ربات تلگرام اینستاگرام
 * 
 * این فایل شامل توابع پردازش دستورات کاربری است
 */

require_once 'config.php';
require_once 'functions.php';

/**
 * پردازش دستور شروع (/start)
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 */
function handleStart($chatId, $userId) {
    global $messages;
    
    // ارسال پیام خوش‌آمدگویی با کیبورد اصلی
    sendMessage($chatId, $messages['welcome'], getMainKeyboard());
}

/**
 * پردازش دستور راهنما (/help)
 * 
 * @param int $chatId شناسه چت
 */
function handleHelp($chatId) {
    global $messages;
    
    // ارسال پیام راهنما
    sendMessage($chatId, $messages['help']);
}

/**
 * پردازش دستور اطلاعات کاربر (/user)
 * 
 * @param int $chatId شناسه چت
 * @param string $username نام کاربری اینستاگرام
 */
function handleUserInfo($chatId, $username) {
    global $messages;
    
    // بررسی صحت نام کاربری
    if (empty($username)) {
        sendMessage($chatId, "لطفاً نام کاربری اینستاگرام را وارد کنید.\nمثال: /user instagram");
        return;
    }
    
    // حذف @ از ابتدای نام کاربری (اگر وجود داشته باشد)
    $username = str_replace('@', '', $username);
    
    // اعتبارسنجی نام کاربری
    if (!validateInstagramUsername($username)) {
        sendErrorMessage($chatId, 'invalid_input');
        return;
    }
    
    // ارسال پیام در حال پردازش
    $processingMessage = sendMessage($chatId, "در حال دریافت اطلاعات پروفایل <b>$username</b>...");
    
    // فراخوانی API اینستاگرام - تلاش با type های مختلف
    $response = callInstagramAPI('info', ['username' => $username]);
    
    // اگر موفق نبود، با profile امتحان می‌کنیم
    if (!$response['success'] || (isset($response['data']['ok']) && !$response['data']['ok'])) {
        $response = callInstagramAPI('profile', ['username' => $username]);
    }
    
    // بررسی پاسخ API
    if (!$response['success']) {
        sendErrorMessage($chatId, 'error');
        return;
    }
    
    $data = $response['data'];
    
    // بررسی وجود داده - پشتیبانی از فرمت‌های مختلف
    $profileInfo = null;
    
    if (isset($data['status']) && $data['status'] === 'ok' && isset($data['result'])) {
        $profileInfo = $data['result'];
    } elseif (isset($data['result']) && $data['result'] === 'ok' && isset($data['data'])) {
        $profileInfo = $data['data'];
    } elseif (isset($data['user'])) {
        $profileInfo = $data['user'];
    } elseif (isset($data['username'])) {
        $profileInfo = $data;
    }
    
    if (!$profileInfo) {
        sendErrorMessage($chatId, 'not_found');
        return;
    }
    
    // ساخت متن پیام
    $caption = "📊 <b>اطلاعات پروفایل</b> @{$username}\n\n";
    
    if (isset($profileInfo['full_name']) && !empty($profileInfo['full_name'])) {
        $caption .= "👤 <b>نام:</b> {$profileInfo['full_name']}\n";
    }
    
    if (isset($profileInfo['biography']) && !empty($profileInfo['biography'])) {
        $caption .= "📝 <b>بیوگرافی:</b> {$profileInfo['biography']}\n";
    }
    
    $caption .= "\n📊 <b>آمار:</b>\n";
    
    // پشتیبانی از فرمت‌های مختلف آمار
    $followerCount = 0;
    $followingCount = 0;
    $mediaCount = 0;
    
    if (isset($profileInfo['follower_count'])) {
        $followerCount = $profileInfo['follower_count'];
    } elseif (isset($profileInfo['edge_followed_by']['count'])) {
        $followerCount = $profileInfo['edge_followed_by']['count'];
    }
    
    if (isset($profileInfo['following_count'])) {
        $followingCount = $profileInfo['following_count'];
    } elseif (isset($profileInfo['edge_follow']['count'])) {
        $followingCount = $profileInfo['edge_follow']['count'];
    }
    
    if (isset($profileInfo['media_count'])) {
        $mediaCount = $profileInfo['media_count'];
    } elseif (isset($profileInfo['edge_owner_to_timeline_media']['count'])) {
        $mediaCount = $profileInfo['edge_owner_to_timeline_media']['count'];
    }
    
    $caption .= "👥 <b>فالوورها:</b> " . number_format($followerCount) . "\n";
    $caption .= "👣 <b>فالووینگ:</b> " . number_format($followingCount) . "\n";
    $caption .= "📷 <b>پست‌ها:</b> " . number_format($mediaCount) . "\n";
    
    // اضافه کردن اطلاعات بیشتر
    if (isset($profileInfo['is_private']) && $profileInfo['is_private']) {
        $caption .= "🔒 این اکانت <b>خصوصی</b> است.\n\n";
    }
    
    if (isset($profileInfo['is_verified']) && $profileInfo['is_verified']) {
        $caption .= "✅ این اکانت <b>تأیید شده</b> است.\n\n";
    }
    
    // اضافه کردن دکمه‌های اینلاین
    $inlineKeyboard = [
        'inline_keyboard' => [
            [
                ['text' => '📸 مشاهده استوری‌ها', 'callback_data' => "story:{$username}"]
            ]
        ]
    ];
    
    // ارسال عکس پروفایل با اطلاعات
    if (isset($profileInfo['profile_pic_url_hd'])) {
        sendPhoto($chatId, $profileInfo['profile_pic_url_hd'], $caption, $inlineKeyboard);
    } else {
        sendMessage($chatId, $caption, $inlineKeyboard);
    }
}

/**
 * پردازش دستور اطلاعات پست (/post)
 * 
 * @param int $chatId شناسه چت
 * @param string $url آدرس پست اینستاگرام
 */
function handlePostInfo($chatId, $url) {
    global $messages;
    
    // بررسی صحت URL
    if (empty($url)) {
        sendMessage($chatId, "لطفاً لینک پست اینستاگرام را وارد کنید.\nمثال: /post https://www.instagram.com/p/ABC123/");
        return;
    }
    
    // اعتبارسنجی URL
    if (!validateInstagramUrl($url)) {
        sendErrorMessage($chatId, 'invalid_input');
        return;
    }
    
    // ارسال پیام در حال پردازش
    $processingMessage = sendMessage($chatId, "در حال دریافت اطلاعات پست...");
    
    // فراخوانی API اینستاگرام
    $response = callInstagramAPI('post', ['url' => $url]);
    
    // بررسی پاسخ API
    if (!$response['success']) {
        sendErrorMessage($chatId, 'error');
        return;
    }
    
    $data = $response['data'];
    
    // بررسی وجود داده
    if (!isset($data['result']) || $data['result'] !== 'ok') {
        sendErrorMessage($chatId, 'not_found');
        return;
    }
    
    // استخراج اطلاعات پست
    $postInfo = $data['data'];
    
    // ساخت متن کپشن
    $caption = "📥 <b>اطلاعات پست</b>\n\n";
    
    // اضافه کردن نام کاربری
    if (isset($postInfo['owner']['username'])) {
        $caption .= "👤 <b>کاربر:</b> @{$postInfo['owner']['username']}\n\n";
    }
    
    // اضافه کردن کپشن پست
    if (isset($postInfo['edge_media_to_caption']['edges'][0]['node']['text'])) {
        $postCaption = $postInfo['edge_media_to_caption']['edges'][0]['node']['text'];
        // محدود کردن طول کپشن
        if (mb_strlen($postCaption) > 300) {
            $postCaption = mb_substr($postCaption, 0, 300) . '...';
        }
        $caption .= "📝 <b>کپشن:</b> {$postCaption}\n\n";
    }
    
    // اضافه کردن آمار
    $caption .= "❤️ <b>لایک‌ها:</b> " . number_format($postInfo['edge_media_preview_like']['count']) . "\n";
    $caption .= "💬 <b>کامنت‌ها:</b> " . number_format($postInfo['edge_media_to_comment']['count']) . "\n\n";
    
    // بررسی نوع پست (تک رسانه یا گالری)
    $isGallery = isset($postInfo['edge_sidecar_to_children']);
    
    if ($isGallery) {
        $caption .= "🖼 این پست شامل <b>چندین رسانه</b> است.\n";
        $mediaItems = $postInfo['edge_sidecar_to_children']['edges'];
        
        // ارسال اولین رسانه با کپشن
        $firstMedia = $mediaItems[0]['node'];
        $mediaUrl = $firstMedia['display_url'];
        $isVideo = isset($firstMedia['is_video']) && $firstMedia['is_video'];
        
        // دکمه‌های اینلاین برای دانلود
        $inlineKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬇️ دانلود این رسانه', 'callback_data' => "download:{$url}"]
                ],
                [
                    ['text' => '👤 پروفایل کاربر', 'callback_data' => "user:{$postInfo['owner']['username']}"]
                ]
            ]
        ];
        
        if ($isVideo) {
            sendMessage($chatId, $caption . "\n\n🎬 این یک ویدیو است. برای دانلود از دکمه زیر استفاده کنید.", $inlineKeyboard);
        } else {
            sendPhoto($chatId, $mediaUrl, $caption, $inlineKeyboard);
        }
        
    } else {
        // پست تک رسانه
        $isVideo = isset($postInfo['is_video']) && $postInfo['is_video'];
        $mediaUrl = $postInfo['display_url'];
        
        // دکمه‌های اینلاین برای دانلود
        $inlineKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬇️ دانلود این رسانه', 'callback_data' => "download:{$url}"]
                ],
                [
                    ['text' => '👤 پروفایل کاربر', 'callback_data' => "user:{$postInfo['owner']['username']}"]
                ]
            ]
        ];
        
        if ($isVideo) {
            sendMessage($chatId, $caption . "\n\n🎬 این یک ویدیو است. برای دانلود از دکمه زیر استفاده کنید.", $inlineKeyboard);
        } else {
            sendPhoto($chatId, $mediaUrl, $caption, $inlineKeyboard);
        }
    }
}

/**
 * پردازش دستور اطلاعات استوری (/story)
 * 
 * @param int $chatId شناسه چت
 * @param string $username نام کاربری اینستاگرام
 */
function handleStoryInfo($chatId, $username) {
    global $messages;
    
    // بررسی صحت نام کاربری
    if (empty($username)) {
        sendMessage($chatId, "لطفاً نام کاربری اینستاگرام را وارد کنید.\nمثال: /story instagram");
        return;
    }
    
    // حذف @ از ابتدای نام کاربری (اگر وجود داشته باشد)
    $username = str_replace('@', '', $username);
    
    // اعتبارسنجی نام کاربری
    if (!validateInstagramUsername($username)) {
        sendErrorMessage($chatId, 'invalid_input');
        return;
    }
    
    // ارسال پیام در حال پردازش
    $processingMessage = sendMessage($chatId, "در حال دریافت استوری‌های <b>$username</b>...");
    
    // فراخوانی API اینستاگرام
    $response = callInstagramAPI('story', ['username' => $username]);
    
    // بررسی پاسخ API
    if (!$response['success']) {
        sendErrorMessage($chatId, 'error');
        return;
    }
    
    $data = $response['data'];
    
    // بررسی وجود داده
    if (!isset($data['result']) || $data['result'] !== 'ok') {
        sendMessage($chatId, "⚠️ کاربر <b>$username</b> استوری فعالی ندارد یا استوری‌های او خصوصی است.");
        return;
    }
    
    // استخراج اطلاعات استوری‌ها
    $stories = $data['data'];
    
    if (empty($stories)) {
        sendMessage($chatId, "⚠️ کاربر <b>$username</b> استوری فعالی ندارد.");
        return;
    }
    
    // ارسال پیام اطلاعات استوری‌ها
    $message = "📹 <b>استوری‌های $username</b>\n\n";
    $message .= "تعداد استوری‌های فعال: <b>" . count($stories) . "</b>\n\n";
    $message .= "برای دانلود هر استوری، روی دکمه مربوطه کلیک کنید:";
    
    // ساخت دکمه‌های اینلاین برای هر استوری
    $inlineKeyboard = ['inline_keyboard' => []];
    
    $counter = 1;
    foreach ($stories as $story) {
        $storyType = isset($story['is_video']) && $story['is_video'] ? '🎬' : '🖼';
        $storyUrl = $story['url'];
        
        // اضافه کردن دکمه برای هر استوری
        $inlineKeyboard['inline_keyboard'][] = [
            ['text' => "$storyType استوری شماره $counter", 'callback_data' => "download:$storyUrl"]
        ];
        
        $counter++;
    }
    
    // اضافه کردن دکمه پروفایل کاربر
    $inlineKeyboard['inline_keyboard'][] = [
        ['text' => '👤 مشاهده پروفایل', 'callback_data' => "user:$username"]
    ];
    
    // ارسال پیام با دکمه‌های اینلاین
    sendMessage($chatId, $message, $inlineKeyboard);
}

/**
 * پردازش دستور دانلود مدیا (/download)
 * 
 * @param int $chatId شناسه چت
 * @param string $url آدرس مدیا
 */
function handleDownload($chatId, $url) {
    global $messages;
    
    // بررسی صحت URL
    if (empty($url)) {
        sendMessage($chatId, "لطفاً لینک مدیای اینستاگرام را وارد کنید.\nمثال: /download https://www.instagram.com/p/ABC123/");
        return;
    }
    
    // ارسال پیام در حال پردازش
    $processingMessage = sendMessage($chatId, "در حال دانلود مدیا...");
    
    // تشخیص نوع URL (پست یا استوری)
    $isDirectUrl = (strpos($url, 'instagram.com') === false);
    
    if ($isDirectUrl) {
        // URL مستقیم مدیا
        $mediaUrl = $url;
        $isVideo = (strpos($url, '.mp4') !== false);
        
        if ($isVideo) {
            sendVideo($chatId, $mediaUrl, "🎬 ویدیو درخواستی شما");
        } else {
            sendPhoto($chatId, $mediaUrl, "🖼 تصویر درخواستی شما");
        }
    } else {
        // URL پست اینستاگرام
        if (!validateInstagramUrl($url)) {
            sendErrorMessage($chatId, 'invalid_input');
            return;
        }
        
        // فراخوانی API اینستاگرام
        $response = callInstagramAPI('post', ['url' => $url]);
        
        // بررسی پاسخ API
        if (!$response['success']) {
            sendErrorMessage($chatId, 'error');
            return;
        }
        
        $data = $response['data'];
        
        // بررسی وجود داده
        if (!isset($data['result']) || $data['result'] !== 'ok') {
            sendErrorMessage($chatId, 'not_found');
            return;
        }
        
        // استخراج اطلاعات مدیا
        $mediaInfo = $data['data'];
        
        // بررسی نوع مدیا (تک رسانه یا گالری)
        if (isset($mediaInfo['edge_sidecar_to_children'])) {
            // پست گالری
            $mediaItems = $mediaInfo['edge_sidecar_to_children']['edges'];
            
            $message = "🖼 این پست شامل <b>" . count($mediaItems) . "</b> رسانه است.\n\n";
            $message .= "برای دانلود هر رسانه، روی دکمه مربوطه کلیک کنید:";
            
            // ساخت دکمه‌های اینلاین برای هر رسانه
            $inlineKeyboard = ['inline_keyboard' => []];
            
            $counter = 1;
            foreach ($mediaItems as $item) {
                $media = $item['node'];
                $isItemVideo = isset($media['is_video']) && $media['is_video'];
                $mediaType = $isItemVideo ? '🎬' : '🖼';
                $mediaUrl = $isItemVideo ? $media['video_url'] : $media['display_url'];
                
                // اضافه کردن دکمه برای هر رسانه
                $inlineKeyboard['inline_keyboard'][] = [
                    ['text' => "$mediaType رسانه شماره $counter", 'callback_data' => "download:$mediaUrl"]
                ];
                
                $counter++;
            }
            
            // ارسال پیام با دکمه‌های اینلاین
            sendMessage($chatId, $message, $inlineKeyboard);
            
        } else {
            // پست تک رسانه
            $isVideo = isset($mediaInfo['is_video']) && $mediaInfo['is_video'];
            $mediaUrl = $isVideo ? $mediaInfo['video_url'] : $mediaInfo['display_url'];
            
            if ($isVideo) {
                sendVideo($chatId, $mediaUrl, "🎬 ویدیو درخواستی شما");
            } else {
                sendPhoto($chatId, $mediaUrl, "🖼 تصویر درخواستی شما");
            }
        }
    }
}

/**
 * پردازش دستور اتصال (/connect)
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 */
function handleConnect($chatId, $userId) {
    // این قسمت می‌تواند برای اتصال کاربر به حساب کاربری یا ذخیره تنظیمات شخصی استفاده شود
    sendMessage($chatId, "🔄 این قابلیت در حال حاضر در دسترس نیست. لطفاً بعداً دوباره امتحان کنید.");
}