<?php

function load_links() {
    $file = LINKS_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_links($links) {
    $file = LINKS_FILE;
    file_put_contents($file, json_encode($links, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function load_users() {
    $file = USERS_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_users($users) {
    $file = USERS_FILE;
    file_put_contents($file, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function load_states() {
    $file = STATES_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_states($states) {
    $file = STATES_FILE;
    file_put_contents($file, json_encode($states, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function set_user_state($user_id, $state) {
    $states = load_states();
    $states[$user_id] = $state;
    save_states($states);
}

function get_user_state($user_id) {
    $states = load_states();
    return isset($states[$user_id]) ? $states[$user_id] : null;
}

function add_user($user_id, $username) {
    $users = load_users();
    if (!isset($users[$user_id])) {
        $users[$user_id] = [
            'username' => $username,
            'joined_at' => date('Y-m-d H:i:s')
        ];
        save_users($users);
    }
}

function generate_unique_id($user_id) {
    $encoded_id = base64_encode($user_id);
    return $encoded_id . '_' . bin2hex(random_bytes(4));
}

function load_channels() {
    $file = __DIR__ . '/channels.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['required_channels' => []]));
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_channels($channels) {
    $file = __DIR__ . '/channels.json';
    file_put_contents($file, json_encode($channels, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function add_required_channel($channel_data) {
    $channels = load_channels();
    
    // اطمینان از وجود ساختار جدید
    if (!isset($channels['required_channels'])) {
        $channels['required_channels'] = [];
    }
    
    // تبدیل ساختار قدیمی به جدید (یکبار)
    if (!empty($channels['required_channels']) && is_string($channels['required_channels'][0])) {
        $old_channels = $channels['required_channels'];
        $channels['required_channels'] = [];
        foreach ($old_channels as $old_channel) {
            $channels['required_channels'][] = [
                'id' => $old_channel,
                'name' => '@' . $old_channel,
                'link' => 'https://t.me/' . $old_channel
            ];
        }
    }
    
    // پارس کردن اطلاعات ورودی
    if (is_string($channel_data)) {
        // فرمت قدیمی - فقط username
        $channel_id = trim(str_replace('@', '', $channel_data));
        $channel_info = [
            'id' => $channel_id,
            'name' => '@' . $channel_id,
            'link' => 'https://t.me/' . $channel_id
        ];
    } else {
        $channel_info = $channel_data;
    }
    
    // چک کردن تکراری نبودن
    foreach ($channels['required_channels'] as $existing) {
        if ($existing['id'] === $channel_info['id']) {
    return false;
}
    }
    
    $channels['required_channels'][] = $channel_info;
    save_channels($channels);
    return true;
}

function remove_required_channel($channel_id) {
    $channels = load_channels();
    
    // نرمال‌سازی ورودی: حذف @ و فضای خالی
    $channel_id = trim(str_replace('@', '', $channel_id));
    
    // پشتیبانی از ساختار قدیمی
    if (!empty($channels['required_channels']) && is_string($channels['required_channels'][0])) {
        $key = array_search($channel_id, $channels['required_channels']);
    if ($key !== false) {
        unset($channels['required_channels'][$key]);
        $channels['required_channels'] = array_values($channels['required_channels']);
        save_channels($channels);
        return true;
        }
        return false;
    }
    
    // ساختار جدید - بررسی ID با حالت‌های مختلف
    foreach ($channels['required_channels'] as $key => $channel) {
        $stored_id = $channel['id'];
        
        // مقایسه مستقیم (با تبدیل نوع)
        if ($stored_id == $channel_id) {
            unset($channels['required_channels'][$key]);
            $channels['required_channels'] = array_values($channels['required_channels']);
            save_channels($channels);
            return true;
        }
        
        // اگر ورودی عددی است، امتحان کن با علامت منفی
        if (is_numeric($channel_id)) {
            // اگر کاربر علامت منفی رو فراموش کرده
            if ($stored_id == "-$channel_id") {
                unset($channels['required_channels'][$key]);
                $channels['required_channels'] = array_values($channels['required_channels']);
                save_channels($channels);
                return true;
            }
            // یا اگر کاربر علامت منفی زده ولی ID در دیتابیس بدون منفی هست
            if (strpos($channel_id, '-') === 0 && $stored_id == substr($channel_id, 1)) {
                unset($channels['required_channels'][$key]);
                $channels['required_channels'] = array_values($channels['required_channels']);
                save_channels($channels);
                return true;
            }
        }
    }
    return false;
}

function check_user_membership($user_id) {
    // اگر کاربر اشتراک فعال دارد، نیازی به عضویت اجباری نیست
    if (has_active_subscription($user_id)) {
        return true;
    }

    $channels = load_channels();
    if (empty($channels['required_channels'])) {
        return true;
    }

    $not_joined = [];
    foreach ($channels['required_channels'] as $channel) {
        // پشتیبانی از ساختار قدیمی و جدید
        if (is_string($channel)) {
            $channel_id = '@' . $channel;
            $channel_name = $channel;
        } else {
            $channel_id = $channel['id'];
            // اگر شناسه عددی است، @ نمی‌گذاریم
            if (!is_numeric($channel_id) && strpos($channel_id, '-') !== 0) {
                $channel_id = '@' . $channel_id;
            }
            $channel_name = $channel['name'] ?? $channel_id;
        }
        
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/getChatMember";
        $data = [
            'chat_id' => $channel_id,
            'user_id' => $user_id
        ];
        
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 5,
                    'ignore_errors' => true
                ]
            ]);
            
            $response = file_get_contents($url . '?' . http_build_query($data), false, $context);
            if ($response === false) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: خطای اتصال");
                continue;
            }
            
            $result = json_decode($response, true);
            if (!$result || !isset($result['ok'])) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: پاسخ نامعتبر");
                continue;
            }
            
            if (!$result['ok']) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: " . ($result['description'] ?? 'خطای ناشناخته'));
                continue;
            }
            
            if (!in_array($result['result']['status'], ['member', 'administrator', 'creator'])) {
                $not_joined[] = is_string($channel) ? $channel : $channel['id'];
            }
        } catch (Exception $e) {
            error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: " . $e->getMessage());
            continue;
        }
    }
    
    return empty($not_joined);
}

function add_link($unique_id, $original_link, $user_id) {
    $links = load_links();
    $links[$unique_id] = [
        'original_link' => $original_link,
        'user_id' => $user_id,
        'created_at' => date('Y-m-d H:i:s')
    ];
    save_links($links);
}

function get_link($unique_id) {
    $links = load_links();
    return isset($links[$unique_id]) ? $links[$unique_id]['original_link'] : null;
}

function send_message($chat_id, $text, $reply_markup = null, $disable_web_page_preview = true) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => $disable_web_page_preview
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    $response = file_get_contents($url . '?' . http_build_query($data));
    $result = json_decode($response, true);
    return $result['ok'] ? $result['result'] : null;
}

function is_admin($user_id) {
    return in_array($user_id, ADMIN_IDS);
}

function get_stats() {
    $links = load_links();
    $users = load_users();
    $user_count = count($users);
    $link_count = count($links);
    return [
        'user_count' => $user_count,
        'link_count' => $link_count
    ];
}

function broadcast_message($message, $chat_id = null, $message_id = null) {
    $users = load_users();
    $batch_size = 100; // تعداد ارسال در هر بچ
    $total_users = count($users);
    $processed_file = __DIR__ . '/broadcast_processed.json';
    
    // بارگذاری لیست کاربران پردازش شده
    if (file_exists($processed_file)) {
        $processed_data = json_decode(file_get_contents($processed_file), true) ?: [];
    } else {
        $processed_data = [];
    }

    // اطمینان از وجود همه کلیدهای مورد نیاز
    $processed_data = array_merge([
        'message_id' => null,
        'processed_users' => [],
        'total' => 0,
        'success' => 0,
        'failed' => 0,
        'status_chat_id' => null,
        'status_message_id' => null
    ], $processed_data);

    // اگر این یک پیام جدید است یا اطلاعات ناقص است
    if ($processed_data['message_id'] !== $message || 
        $processed_data['status_chat_id'] !== $chat_id || 
        $processed_data['status_message_id'] !== $message_id) {
        
        $processed_data = [
            'message_id' => $message,
            'processed_users' => [],
            'total' => $total_users,
            'success' => 0,
            'failed' => 0,
            'status_chat_id' => $chat_id,
            'status_message_id' => $message_id
        ];
    }

    $current_batch = 0;
    foreach ($users as $user_id => $user) {
        // اگر کاربر قبلاً پیام را دریافت کرده، رد کن
        if (in_array($user_id, $processed_data['processed_users'])) {
            continue;
        }

        // محدودیت تعداد در هر بچ
        if ($current_batch >= $batch_size) {
            break;
        }

        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
        $data = [
            'chat_id' => $user_id,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $response = @file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        
        if ($result && $result['ok']) {
            $processed_data['success']++;
        } else {
            $processed_data['failed']++;
            // اگر کاربر ربات را بلاک کرده یا حذف کرده
            if (isset($result['error_code']) && in_array($result['error_code'], [403, 400])) {
                unset($users[$user_id]);
                save_users($users);
            }
        }

        // اضافه کردن کاربر به لیست پردازش شده
        $processed_data['processed_users'][] = $user_id;
        $current_batch++;
    }

    // ذخیره وضعیت پردازش
    file_put_contents($processed_file, json_encode($processed_data, JSON_PRETTY_PRINT));

    // اگر پنل مدیریت باز است، آمار را برگردان
    if ($processed_data['status_chat_id'] && $processed_data['status_message_id']) {
        $remaining = $total_users - count($processed_data['processed_users']);
        $progress_text = "📊 وضعیت ارسال همگانی:\n\n";
        $progress_text .= "👥 تعداد افراد ارسالی: " . count($processed_data['processed_users']) . "\n";
        $progress_text .= "⏱ زمان تخمینی باقیمانده: " . ceil($remaining / $batch_size) . " دقیقه\n\n";
        $progress_text .= "✅ موفق: " . $processed_data['success'] . "\n";
        $progress_text .= "❌ ناموفق: " . $processed_data['failed'] . "\n";
        
        $reply_markup = null;
        if ($remaining > 0) {
            $reply_markup = [
                'inline_keyboard' => [
                    [['text' => '🔄 بروزرسانی آمار', 'callback_data' => 'refresh_broadcast_stats']]
                ]
            ];
        }
        
        edit_message($processed_data['status_chat_id'], $processed_data['status_message_id'], $progress_text, $reply_markup);
    }

    // اگر همه کاربران پردازش شده‌اند، فایل را پاک کن
    if (count($processed_data['processed_users']) >= $total_users) {
        @unlink($processed_file);
        return true;
    }

    return false;
}

function forward_message_to_all($from_chat_id, $message_id, $chat_id = null, $status_message_id = null) {
    $users = load_users();
    $batch_size = 100; // تعداد ارسال در هر بچ
    $total_users = count($users);
    $processed_file = __DIR__ . '/forward_processed.json';
    
    // بارگذاری لیست کاربران پردازش شده
    if (file_exists($processed_file)) {
        $processed_data = json_decode(file_get_contents($processed_file), true) ?: [];
    } else {
        $processed_data = [];
    }

    // اطمینان از وجود همه کلیدهای مورد نیاز
    $processed_data = array_merge([
        'message_id' => null,
        'processed_users' => [],
        'total' => 0,
        'success' => 0,
        'failed' => 0,
        'status_chat_id' => null,
        'status_message_id' => null
    ], $processed_data);

    $message_key = $from_chat_id . ':' . $message_id;

    // اگر این یک پیام جدید است یا اطلاعات ناقص است
    if ($processed_data['message_id'] !== $message_key || 
        $processed_data['status_chat_id'] !== $chat_id || 
        $processed_data['status_message_id'] !== $status_message_id) {
        
        $processed_data = [
            'message_id' => $message_key,
            'processed_users' => [],
            'total' => $total_users,
            'success' => 0,
            'failed' => 0,
            'status_chat_id' => $chat_id,
            'status_message_id' => $status_message_id
        ];
    }

    $current_batch = 0;
    foreach ($users as $user_id => $user) {
        // اگر کاربر قبلاً پیام را دریافت کرده، رد کن
        if (in_array($user_id, $processed_data['processed_users'])) {
            continue;
        }

        // محدودیت تعداد در هر بچ
        if ($current_batch >= $batch_size) {
            break;
        }

        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/forwardMessage";
        $data = [
            'chat_id' => $user_id,
            'from_chat_id' => $from_chat_id,
            'message_id' => $message_id
        ];
        
        $response = @file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        
        if ($result && $result['ok']) {
            $processed_data['success']++;
        } else {
            $processed_data['failed']++;
            // اگر کاربر ربات را بلاک کرده یا حذف کرده
            if (isset($result['error_code']) && in_array($result['error_code'], [403, 400])) {
                unset($users[$user_id]);
                save_users($users);
            }
        }

        // اضافه کردن کاربر به لیست پردازش شده
        $processed_data['processed_users'][] = $user_id;
        $current_batch++;
    }

    // ذخیره وضعیت پردازش
    file_put_contents($processed_file, json_encode($processed_data, JSON_PRETTY_PRINT));

    // اگر پنل مدیریت باز است، آمار را برگردان
    if ($processed_data['status_chat_id'] && $processed_data['status_message_id']) {
        $remaining = $total_users - count($processed_data['processed_users']);
        $progress_text = "📊 وضعیت فوروارد همگانی:\n\n";
        $progress_text .= "👥 تعداد افراد ارسالی: " . count($processed_data['processed_users']) . "\n";
        $progress_text .= "⏱ زمان تخمینی باقیمانده: " . ceil($remaining / $batch_size) . " دقیقه\n\n";
        $progress_text .= "✅ موفق: " . $processed_data['success'] . "\n";
        $progress_text .= "❌ ناموفق: " . $processed_data['failed'] . "\n";
        
        $reply_markup = null;
        if ($remaining > 0) {
            $reply_markup = [
                'inline_keyboard' => [
                    [['text' => '🔄 بروزرسانی آمار', 'callback_data' => 'refresh_forward_stats']]
                ]
            ];
        }
        
        edit_message($processed_data['status_chat_id'], $processed_data['status_message_id'], $progress_text, $reply_markup);
    }

    // اگر همه کاربران پردازش شده‌اند، فایل را پاک کن
    if (count($processed_data['processed_users']) >= $total_users) {
        @unlink($processed_file);
        return true;
    }

    return false;
}

function delete_message($chat_id, $message_id) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/deleteMessage";
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id
    ];
    
    // Use POST method instead of GET for more reliability
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query($data),
            'ignore_errors' => true
        ]
    ];
    
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        error_log(sprintf(
            "خطا در حذف پیام %d در چت %s: خطای اتصال",
            $message_id,
            $chat_id
        ));
        return false;
    }
    
    $result = json_decode($response, true);
    
    // اگر پیام قبلاً حذف شده یا پیدا نشد، موفقیت آمیز در نظر بگیریم
    if (!$result['ok'] && isset($result['error_code']) && $result['error_code'] === 400) {
        error_log(sprintf(
            "نتیجه حذف پیام %d در چت %s: %s",
            $message_id,
            $chat_id,
            $result['description'] ?? 'خطای ناشناخته'
        ));
        return true;
    }
    
    return $result['ok'] ?? false;
}

function send_temporary_message($chat_id, $text, $reply_markup = null, $delete_after = null) {
    // اگر زمان حذف مشخص نشده، از تنظیمات استفاده کن
    if ($delete_after === null) {
        $delete_after = get_auto_delete_time();
    }
    
    $message = send_message($chat_id, $text, $reply_markup);
    if ($message) {
        // Store deletion info
        $delete_at = time() + $delete_after;
        register_deletion($chat_id, $message['message_id'], $delete_at);
        
        // Schedule immediate background deletion
        schedule_deletion($chat_id, $message['message_id'], $delete_after);
    }
    return $message;
}

function schedule_deletion($chat_id, $message_id, $delay) {
    // Create a background process to handle deletion
    $cmd = sprintf(
        'curl --max-time 10 -s -X POST "https://api.telegram.org/bot%s/deleteMessage" -d "chat_id=%s&message_id=%d" > /dev/null 2>&1',
        BOT_TOKEN,
        $chat_id,
        $message_id
    );
    
    // Use different methods based on OS
    if (PHP_OS === 'WINNT') {
        // Windows
        pclose(popen(sprintf('start /B php -r "sleep(%d); %s;"', $delay, $cmd), 'r'));
    } else {
        // Linux/Unix
        shell_exec(sprintf('(sleep %d; %s) > /dev/null 2>&1 &', $delay, $cmd));
        
        // Also schedule through at command as backup
        $at_time = date('H:i', time() + $delay);
        $at_cmd = sprintf('echo "%s" | at %s > /dev/null 2>&1', $cmd, $at_time);
        shell_exec($at_cmd);
    }
}

function register_deletion($chat_id, $message_id, $delete_at) {
    $file = __DIR__ . '/message_deletions.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['deletions' => []]));
    }
    
    $data = json_decode(file_get_contents($file), true) ?: ['deletions' => []];
    
    // Remove any existing deletion for this message
    $data['deletions'] = array_filter($data['deletions'], function($item) use ($chat_id, $message_id) {
        return !($item['chat_id'] == $chat_id && $item['message_id'] == $message_id);
    });
    
    // Add new deletion
    $data['deletions'][] = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'delete_at' => $delete_at,
        'created_at' => time()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

function cleanup_messages() {
    $file = __DIR__ . '/message_deletions.json';
    if (!file_exists($file)) {
        return;
    }
    
    $data = json_decode(file_get_contents($file), true) ?: ['deletions' => []];
    $now = time();
    $remaining = [];
    $modified = false;
    
    foreach ($data['deletions'] as $deletion) {
        if ($deletion['delete_at'] <= $now) {
            // Try to delete the message
            $result = delete_message($deletion['chat_id'], $deletion['message_id']);
            if ($result) {
                $modified = true;
                continue;
            }
            
            // اگر حذف ناموفق بود و پیام بیش از 5 دقیقه از موعد حذفش گذشته
            if ($now - $deletion['delete_at'] > 300) {
                $modified = true;
                continue;
            }
        }
        $remaining[] = $deletion;
    }
    
    if ($modified) {
        $data['deletions'] = $remaining;
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
    }
}

function edit_message($chat_id, $message_id, $text, $reply_markup = null) {
    static $last_message = [];
    $message_key = $chat_id . '_' . $message_id;
    
    // ساخت محتوای پیام برای مقایسه
    $current_content = [
        'text' => $text,
        'reply_markup' => $reply_markup ? json_encode($reply_markup) : null
    ];
    
    // اگر پیام تغییر نکرده، آپدیت نکن
    if (isset($last_message[$message_key]) && 
        $last_message[$message_key]['text'] === $current_content['text'] && 
        $last_message[$message_key]['reply_markup'] === $current_content['reply_markup']) {
        return true;
    }
    
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/editMessageText";
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }

    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query($data),
            'ignore_errors' => true
        ]
    ];
    
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        error_log("خطا در ویرایش پیام: خطای اتصال");
        return false;
    }
    
    $result = json_decode($response, true);
    
    if (!$result['ok']) {
        // اگر پیام تغییر نکرده، خطا نیست
        if (isset($result['description']) && strpos($result['description'], 'message is not modified') !== false) {
            $last_message[$message_key] = $current_content;
            return true;
        }
        error_log("خطا در ویرایش پیام: " . ($result['description'] ?? 'خطای ناشناخته'));
        return false;
    }
    
    // ذخیره آخرین محتوای پیام
    $last_message[$message_key] = $current_content;
    return $result;
}

function answer_callback_query($callback_query_id, $text = null, $show_alert = false) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/answerCallbackQuery";
    $data = [
        'callback_query_id' => $callback_query_id,
        'show_alert' => $show_alert
    ];
    if ($text) {
        $data['text'] = $text;
    }
    $response = file_get_contents($url . '?' . http_build_query($data));
    return json_decode($response, true);
}

function get_links_paginated($page = 1, $per_page = 5) {
    $links = load_links();
    $total = count($links);
    $total_pages = ceil($total / $per_page);
    $page = max(1, min($page, $total_pages));
    
    $start = ($page - 1) * $per_page;
    $paginated = array_slice($links, $start, $per_page, true);
    
    return [
        'links' => $paginated,
        'total' => $total,
        'per_page' => $per_page,
        'current_page' => $page,
        'total_pages' => $total_pages
    ];
}

function get_detailed_stats() {
    $stats = get_stats();
    $users = load_users();
    $links = load_links();
    
    // آمار کاربران فعال در 24 ساعت گذشته
    $active_users = 0;
    $now = time();
    foreach ($users as $user) {
        if (isset($user['last_activity']) && ($now - strtotime($user['last_activity'])) <= 86400) {
            $active_users++;
        }
    }
    
    // آمار لینک‌های ایجاد شده در 24 ساعت گذشته
    $recent_links = 0;
    foreach ($links as $link) {
        if ($now - strtotime($link['created_at']) <= 86400) {
            $recent_links++;
        }
    }
    
    return [
        'total_users' => $stats['user_count'],
        'total_links' => $stats['link_count'],
        'active_users_24h' => $active_users,
        'new_links_24h' => $recent_links
    ];
}

function get_channel_keyboard() {
    $channels = load_channels();
    $keyboard = [];
    
    foreach ($channels['required_channels'] as $channel) {
        // پشتیبانی از ساختار قدیمی و جدید
        if (is_string($channel)) {
            $keyboard[] = [
                ['text' => "@$channel", 'url' => "https://t.me/$channel"]
            ];
        } else {
            $text = isset($channel['name']) ? $channel['name'] : "کانال";
            $url = $channel['link'] ?? "https://t.me/" . $channel['id'];
            $keyboard[] = [
                ['text' => $text, 'url' => $url]
            ];
        }
    }
    
    $keyboard[] = [
        ['text' => "✅ بررسی عضویت", 'callback_data' => 'check_membership']
    ];
    
    return [
        'inline_keyboard' => $keyboard
    ];
}

function get_start_message() {
    $file = __DIR__ . '/start_message.json';
    if (!file_exists($file)) {
        $default_data = [
            'type' => 'text',
            'message' => "👋 سلام! لینک خود را ارسال کنید تا محافظت شود.",
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
        file_put_contents($file, json_encode($default_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default_data;
    }
    $data = json_decode(file_get_contents($file), true);
    
    // پشتیبانی از فرمت قدیمی
    if (isset($data['message']) && is_string($data['message']) && !isset($data['type'])) {
        $old_message = $data['message'];
        $data = [
            'type' => 'text',
            'message' => $old_message,
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    return $data;
}

function save_start_message($message_data) {
    $file = __DIR__ . '/start_message.json';
    
    // اگر رشته ساده است، به ساختار جدید تبدیل کن
    if (is_string($message_data)) {
        $message_data = [
            'type' => 'text',
            'message' => $message_data,
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
    }
    
    return file_put_contents($file, json_encode($message_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function send_start_message($chat_id, $custom_keyboard = null) {
    $start_data = get_start_message();
    
    // اگر نوع پیام متن یا عکس است
    $text = $start_data['message'] ?? "👋 سلام! لینک خود را ارسال کنید تا محافظت شود.";
    $photo = $start_data['photo'] ?? null;
    $inline_buttons = $start_data['inline_buttons'] ?? [];
    
    // ساخت کیبورد شیشه‌ای
    $reply_markup = null;
    if (!empty($inline_buttons)) {
        $keyboard_rows = [];
        foreach ($inline_buttons as $btn) {
            $keyboard_rows[] = [
                ['text' => $btn['text'], 'url' => $btn['url']]
            ];
        }
        $reply_markup = ['inline_keyboard' => $keyboard_rows];
    }
    
    // اگر کیبورد سفارشی داریم و inline button نداریم
    if (!$reply_markup && $custom_keyboard) {
        $reply_markup = $custom_keyboard;
    }
    
    // اگر عکس داریم
    if ($photo) {
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendPhoto";
        $data = [
            'chat_id' => $chat_id,
            'photo' => $photo,
            'caption' => $text,
            'parse_mode' => 'HTML'
        ];
        
        if ($reply_markup) {
            $data['reply_markup'] = json_encode($reply_markup);
        }
        
        $response = file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        return $result['ok'] ? $result['result'] : null;
    }
    
    // فقط متن
    return send_message($chat_id, $text, $reply_markup);
}

function get_forced_join_message() {
    $file = __DIR__ . '/forced_join_message.json';
    if (!file_exists($file)) {
        $default_message = "⚠️ لطفاً ابتدا در کانال‌های زیر عضو شوید:\n\n";
        file_put_contents($file, json_encode(['message' => $default_message], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default_message;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['message'] ?? "⚠️ لطفاً ابتدا در کانال‌های زیر عضو شوید:\n\n";
}

function save_forced_join_message($message) {
    $file = __DIR__ . '/forced_join_message.json';
    return file_put_contents($file, json_encode(['message' => $message], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function get_auto_delete_time() {
    $file = __DIR__ . '/auto_delete_settings.json';
    if (!file_exists($file)) {
        $default_time = 60; // 60 ثانیه (1 دقیقه)
        file_put_contents($file, json_encode(['delete_after' => $default_time], JSON_PRETTY_PRINT));
        return $default_time;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['delete_after'] ?? 60;
}

function save_auto_delete_time($seconds) {
    // محدود کردن بین 10 ثانیه تا 300 ثانیه (5 دقیقه)
    $seconds = max(10, min(300, intval($seconds)));
    $file = __DIR__ . '/auto_delete_settings.json';
    return file_put_contents($file, json_encode(['delete_after' => $seconds], JSON_PRETTY_PRINT));
}

// مدیریت دکمه‌های سفارشی
function load_custom_buttons() {
    $file = __DIR__ . '/custom_buttons.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['buttons' => []], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['buttons'] ?? [];
}

function save_custom_buttons($buttons) {
    $file = __DIR__ . '/custom_buttons.json';
    return file_put_contents($file, json_encode(['buttons' => $buttons], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function add_custom_button($button_data) {
    $buttons = load_custom_buttons();
    
    // ایجاد ID یکتا برای دکمه
    $button_data['id'] = uniqid('btn_');
    $button_data['created_at'] = date('Y-m-d H:i:s');
    
    $buttons[] = $button_data;
    save_custom_buttons($buttons);
    return $button_data['id'];
}

function get_custom_button($button_id) {
    $buttons = load_custom_buttons();
    foreach ($buttons as $button) {
        if ($button['id'] === $button_id) {
            return $button;
        }
    }
    return null;
}

function get_custom_button_by_name($button_name) {
    $buttons = load_custom_buttons();
    foreach ($buttons as $button) {
        if ($button['name'] === $button_name) {
            return $button;
        }
    }
    return null;
}

function delete_custom_button($button_id) {
    $buttons = load_custom_buttons();
    $new_buttons = array_filter($buttons, function($btn) use ($button_id) {
        return $btn['id'] !== $button_id;
    });
    save_custom_buttons(array_values($new_buttons));
    return count($buttons) !== count($new_buttons);
}

function get_custom_buttons_keyboard() {
    $buttons = load_custom_buttons();
    
    $keyboard = [];
    $row = [];
    
    if (!empty($buttons)) {
        foreach ($buttons as $index => $button) {
            $row[] = ['text' => $button['name']];
            
            // هر 2 دکمه یک ردیف
            if (count($row) == 2 || $index == count($buttons) - 1) {
                $keyboard[] = $row;
                $row = [];
            }
        }
    }
    
    // اضافه کردن دکمه‌های کیف پول و اشتراک
    $keyboard[] = [
        ['text' => '💳 کیف پول من'],
        ['text' => '🛒 خرید اشتراک']
    ];
    $keyboard[] = [['text' => 'ℹ️ وضعیت اشتراک']];
    
    return [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}

// -------- Uploader helpers --------
function load_uploads() {
    $file = __DIR__ . '/uploads.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode([], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    $json = file_get_contents($file);
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

function save_uploads($uploads) {
    $file = __DIR__ . '/uploads.json';
    file_put_contents($file, json_encode($uploads, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function create_upload($user_id) {
    $uploads = load_uploads();
    $upload_id = 'up_' . bin2hex(random_bytes(6));
    $uploads[$upload_id] = [
        'id' => $upload_id,
        'user_id' => $user_id,
        'created_at' => date('Y-m-d H:i:s'),
        'items' => []
    ];
    save_uploads($uploads);
    return $upload_id;
}

function add_upload_item($upload_id, $item) {
    $uploads = load_uploads();
    if (!isset($uploads[$upload_id])) return false;
    $uploads[$upload_id]['items'][] = [
        'type' => $item['type'],
        'file_id' => $item['file_id'],
        'caption' => $item['caption'] ?? null,
        'has_spoiler' => $item['has_spoiler'] ?? false,
        'duration' => $item['duration'] ?? null,
        'width' => $item['width'] ?? null,
        'height' => $item['height'] ?? null,
        'thumbnail' => $item['thumbnail'] ?? null
    ];
    save_uploads($uploads);
    return true;
}

function get_upload($upload_id) {
    $uploads = load_uploads();
    return $uploads[$upload_id] ?? null;
}

function delete_upload($upload_id) {
    $uploads = load_uploads();
    if (isset($uploads[$upload_id])) {
        unset($uploads[$upload_id]);
        save_uploads($uploads);
        return true;
    }
    return false;
}

function get_uploads_paginated($page = 1, $per_page = 5) {
    $uploads = load_uploads();
    // مرتب‌سازی نزولی بر اساس تاریخ ایجاد
    uasort($uploads, function($a, $b) { return strtotime($b['created_at']) <=> strtotime($a['created_at']); });
    $total = count($uploads);
    $total_pages = max(1, ceil($total / $per_page));
    $page = max(1, min($page, $total_pages));
    $start = ($page - 1) * $per_page;
    $paginated = array_slice($uploads, $start, $per_page, true);
    return [
        'uploads' => $paginated,
        'total' => $total,
        'per_page' => $per_page,
        'current_page' => $page,
        'total_pages' => $total_pages
    ];
}

function get_media_group_map() {
    $file = __DIR__ . '/media_groups.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['groups' => []], JSON_PRETTY_PRINT));
    }
    $data = json_decode(file_get_contents($file), true);
    return $data ?: ['groups' => []];
}

function save_media_group_map($data) {
    $file = __DIR__ . '/media_groups.json';
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

function get_or_create_upload_for_group($user_id, $media_group_id) {
    $map = get_media_group_map();
    $key = $user_id . ':' . $media_group_id;
    if (!isset($map['groups'][$key])) {
        $upload_id = create_upload($user_id);
        $map['groups'][$key] = [
            'upload_id' => $upload_id,
            'link_sent' => false,
            'updated_at' => time()
        ];
        save_media_group_map($map);
        return [$upload_id, true]; // true => newly created
    } else {
        $map['groups'][$key]['updated_at'] = time();
        save_media_group_map($map);
        return [$map['groups'][$key]['upload_id'], false];
    }
}

function mark_group_link_sent($user_id, $media_group_id) {
    $map = get_media_group_map();
    $key = $user_id . ':' . $media_group_id;
    if (isset($map['groups'][$key])) {
        $map['groups'][$key]['link_sent'] = true;
        save_media_group_map($map);
    }
}

function was_group_link_sent($user_id, $media_group_id) {
    $map = get_media_group_map();
    $key = $user_id . ':' . $media_group_id;
    return isset($map['groups'][$key]) ? (bool)($map['groups'][$key]['link_sent'] ?? false) : false;
}

function get_file_auto_delete_time() {
    $file = __DIR__ . '/file_auto_delete_settings.json';
    if (!file_exists($file)) {
        $default = 0; // 0 = off
        file_put_contents($file, json_encode(['delete_after' => $default], JSON_PRETTY_PRINT));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return isset($data['delete_after']) ? intval($data['delete_after']) : 0;
}

function save_file_auto_delete_time($seconds) {
    // اجازه 0 (خاموش) تا 300 ثانیه
    $seconds = max(0, min(300, intval($seconds)));
    $file = __DIR__ . '/file_auto_delete_settings.json';
    return file_put_contents($file, json_encode(['delete_after' => $seconds], JSON_PRETTY_PRINT));
}

// تنظیمات spoiler
function get_spoiler_enabled() {
    $file = __DIR__ . '/spoiler_settings.json';
    if (!file_exists($file)) {
        $default = false; // پیش‌فرض غیرفعال
        file_put_contents($file, json_encode(['enabled' => $default], JSON_PRETTY_PRINT));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return (bool)($data['enabled'] ?? false);
}

function save_spoiler_enabled($enabled) {
    $file = __DIR__ . '/spoiler_settings.json';
    return file_put_contents($file, json_encode(['enabled' => (bool)$enabled], JSON_PRETTY_PRINT));
}

// تنظیمات متن حذف خودکار
function get_auto_delete_message() {
    $file = __DIR__ . '/auto_delete_message.json';
    if (!file_exists($file)) {
        $default = "⚠️ فایل(ها) بعد از <b>{TIME}</b> حذف خواهند شد.";
        file_put_contents($file, json_encode(['message' => $default], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['message'] ?? "⚠️ فایل(ها) بعد از <b>{TIME}</b> حذف خواهند شد.";
}

function save_auto_delete_message($message) {
    $file = __DIR__ . '/auto_delete_message.json';
    return file_put_contents($file, json_encode(['message' => $message], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function send_media_by_type($chat_id, $type, $file_id, $caption = null, $has_spoiler = false, $duration = null, $width = null, $height = null, $thumbnail = null, $protect_content = false) {
    $endpoint = null;
    $param_name = null;
    switch ($type) {
        case 'photo':
            $endpoint = 'sendPhoto'; $param_name = 'photo'; break;
        case 'video':
            $endpoint = 'sendVideo'; $param_name = 'video'; break;
        case 'document':
            $endpoint = 'sendDocument'; $param_name = 'document'; break;
        case 'audio':
            $endpoint = 'sendAudio'; $param_name = 'audio'; break;
        case 'voice':
            $endpoint = 'sendVoice'; $param_name = 'voice'; break;
        case 'animation':
            $endpoint = 'sendAnimation'; $param_name = 'animation'; break;
        case 'video_note':
            $endpoint = 'sendVideoNote'; $param_name = 'video_note'; break;
        case 'sticker':
            $endpoint = 'sendSticker'; $param_name = 'sticker'; break;
        default:
            // fallback as document
            $endpoint = 'sendDocument'; $param_name = 'document'; break;
    }
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $endpoint;
    $data = [
        'chat_id' => $chat_id,
        $param_name => $file_id,
    ];
    if ($caption && in_array($type, ['photo','video','document','audio','animation'])) {
        $data['caption'] = $caption;
        $data['parse_mode'] = 'HTML';
    }
    // افزودن metadata برای ویدیو (ضروری برای اسپویلر)
    if ($type === 'video') {
        if ($duration !== null) $data['duration'] = intval($duration);
        if ($width !== null) $data['width'] = intval($width);
        if ($height !== null) $data['height'] = intval($height);
        if ($thumbnail !== null) $data['thumbnail'] = $thumbnail;
        $data['supports_streaming'] = true;
    }
    // پشتیبانی از spoiler برای photo و video
    if ($has_spoiler && in_array($type, ['photo', 'video'])) {
        $data['has_spoiler'] = true;
    }
    if (isset($reply_markup) && $reply_markup) {
        $data['reply_markup'] = $reply_markup;
    }
    if ($protect_content) {
        $data['protect_content'] = true;
    }
    
    // استفاده از JSON برای ارسال (بهتر از form-urlencoded برای boolean ها)
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/json',
            'content' => json_encode($data)
        ]
    ];
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    $result = json_decode($response, true);
    return ($result && ($result['ok'] ?? false)) ? $result['result'] : null;
}

function send_media_by_type_with_buttons($chat_id, $type, $file_id, $caption = null, $has_spoiler = false, $reply_markup = null, $duration = null, $width = null, $height = null, $thumbnail = null, $protect_content = false) {
    $endpoint = null;
    $param_name = null;
    switch ($type) {
        case 'photo':
            $endpoint = 'sendPhoto'; $param_name = 'photo'; break;
        case 'video':
            $endpoint = 'sendVideo'; $param_name = 'video'; break;
        case 'document':
            $endpoint = 'sendDocument'; $param_name = 'document'; break;
        case 'audio':
            $endpoint = 'sendAudio'; $param_name = 'audio'; break;
        case 'voice':
            $endpoint = 'sendVoice'; $param_name = 'voice'; break;
        case 'animation':
            $endpoint = 'sendAnimation'; $param_name = 'animation'; break;
        case 'video_note':
            $endpoint = 'sendVideoNote'; $param_name = 'video_note'; break;
        case 'sticker':
            $endpoint = 'sendSticker'; $param_name = 'sticker'; break;
        default:
            // fallback as document
            $endpoint = 'sendDocument'; $param_name = 'document'; break;
    }
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $endpoint;
    $data = [
        'chat_id' => $chat_id,
        $param_name => $file_id,
    ];
    if ($caption && in_array($type, ['photo','video','document','audio','animation'])) {
        $data['caption'] = $caption;
        $data['parse_mode'] = 'HTML';
    }
    // افزودن metadata برای ویدیو (ضروری برای اسپویلر)
    if ($type === 'video') {
        if ($duration !== null) $data['duration'] = intval($duration);
        if ($width !== null) $data['width'] = intval($width);
        if ($height !== null) $data['height'] = intval($height);
        if ($thumbnail !== null) $data['thumbnail'] = $thumbnail;
        $data['supports_streaming'] = true;
    }
    // پشتیبانی از spoiler فقط برای photo و video (طبق مستندات تلگرام)
    if ($has_spoiler && in_array($type, ['photo', 'video'])) {
        $data['has_spoiler'] = true;
    }
    if ($reply_markup) {
        $data['reply_markup'] = $reply_markup;
    }
    if ($protect_content) {
        $data['protect_content'] = true;
    }
    
    // استفاده از JSON برای ارسال (بهتر از form-urlencoded برای boolean ها)
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/json',
            'content' => json_encode($data)
        ]
    ];
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    $result = json_decode($response, true);
    return ($result && ($result['ok'] ?? false)) ? $result['result'] : null;
}

// تنظیمات دکمه‌های تبلیغاتی
function get_ad_buttons() {
    $file = __DIR__ . '/ad_buttons.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['buttons' => []], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return [];
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['buttons'] ?? [];
}

function save_ad_buttons($buttons) {
    $file = __DIR__ . '/ad_buttons.json';
    return file_put_contents($file, json_encode(['buttons' => $buttons], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function add_ad_button($text, $url) {
    $buttons = get_ad_buttons();
    $buttons[] = ['text' => $text, 'url' => $url];
    return save_ad_buttons($buttons);
}

function remove_ad_button($index) {
    $buttons = get_ad_buttons();
    if (isset($buttons[$index])) {
        array_splice($buttons, $index, 1);
        return save_ad_buttons($buttons);
    }
    return false;
}

// سیستم batch upload
function get_batch_upload($user_id) {
    $file = __DIR__ . '/batch_uploads.json';
    if (!file_exists($file)) {
        return null;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data[$user_id] ?? null;
}

function save_batch_upload($user_id, $batch_data) {
    $file = __DIR__ . '/batch_uploads.json';
    $data = [];
    if (file_exists($file)) {
        $data = json_decode(file_get_contents($file), true) ?? [];
    }
    $data[$user_id] = $batch_data;
    return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function clear_batch_upload($user_id) {
    $file = __DIR__ . '/batch_uploads.json';
    if (!file_exists($file)) {
        return true;
    }
    $data = json_decode(file_get_contents($file), true) ?? [];
    if (isset($data[$user_id])) {
        unset($data[$user_id]);
        return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    return true;
}

function add_to_batch($user_id, $item) {
    $batch = get_batch_upload($user_id);
    if (!$batch) {
        $batch = ['items' => [], 'updated_at' => time()];
    }
    $batch['items'][] = $item;
    $batch['updated_at'] = time();
    return save_batch_upload($user_id, $batch);
}

function send_uploaded_files($chat_id, $upload_id) {
    $upload = get_upload($upload_id);
    if (!$upload || empty($upload['items'])) {
        send_message($chat_id, "❌ فایل(ها) یافت نشد یا منقضی شده است.");
        return false;
    }
    $file_delete_after = get_file_auto_delete_time();
    $spoiler_enabled = get_spoiler_enabled();
    $like_enabled = get_like_enabled();
    $view_enabled = get_view_count_enabled();
    $resend_enabled = get_resend_enabled();
    $ad_buttons = get_ad_buttons();
    $forward_lock = $upload['forward_lock'] ?? false; // Check for forward lock
    $sent_any = false;
    
    // ثبت بازدید کاربر
    if ($view_enabled) {
        add_view($upload_id, $chat_id);
    }
    
    // آماده‌سازی reply markup برای دکمه‌های تبلیغاتی
    $reply_markup = null;
    $inline_keyboard = [];
    
    // دکمه‌های تبلیغاتی
    if (!empty($ad_buttons)) {
        foreach ($ad_buttons as $btn) {
            $inline_keyboard[] = [['text' => $btn['text'], 'url' => $btn['url']]];
        }
    }
    
    // دکمه‌های لایک و بازدید
    $interaction_row = [];
    if ($like_enabled) {
        $likes = get_likes($upload_id);
        $like_text = ($likes['likes'] > 0) ? "👍 (" . $likes['likes'] . ")" : "👍 (0)";
        $dislike_text = ($likes['dislikes'] > 0) ? "👎 (" . $likes['dislikes'] . ")" : "👎 (0)";
        $interaction_row[] = ['text' => $like_text, 'callback_data' => 'like_' . $upload_id];
        $interaction_row[] = ['text' => $dislike_text, 'callback_data' => 'dislike_' . $upload_id];
    }
    if ($view_enabled) {
        $views = get_views($upload_id);
        $view_text = "👁 بازدید: " . $views;
        $interaction_row[] = ['text' => $view_text, 'callback_data' => 'views_' . $upload_id];
    }
    if (!empty($interaction_row)) {
        $inline_keyboard[] = $interaction_row;
    }
    
    if (!empty($inline_keyboard)) {
        $reply_markup = ['inline_keyboard' => $inline_keyboard];
    }
    
    // ارسال فایل‌ها
    foreach ($upload['items'] as $idx => $item) {
        // استفاده از اسپویلر: یا از تنظیمات سراسری یا از اسپویلر اصلی فایل
        $file_had_spoiler = $item['has_spoiler'] ?? false;
        $use_spoiler = false;
        
        // فقط برای photo و video اسپویلر کار می‌کند
        if (in_array($item['type'], ['photo', 'video'])) {
            // اگر فایل اصلی اسپویلر داشت یا تنظیمات سراسری فعال باشد
            $use_spoiler = $file_had_spoiler || $spoiler_enabled;
        }
        
        // استخراج metadata برای ویدیو
        $duration = $item['duration'] ?? null;
        $width = $item['width'] ?? null;
        $height = $item['height'] ?? null;
        $thumbnail = $item['thumbnail'] ?? null;
        
        $msg = send_media_by_type_with_buttons($chat_id, $item['type'], $item['file_id'], $item['caption'] ?? null, $use_spoiler, $reply_markup, $duration, $width, $height, $thumbnail, $forward_lock);
        
        // Only schedule deletion if forward lock is NOT enabled
        if ($msg && $file_delete_after > 0 && !$forward_lock) {
            register_deletion($chat_id, $msg['message_id'], time() + $file_delete_after);
            // Also background schedule
            schedule_deletion($chat_id, $msg['message_id'], $file_delete_after);
        }
        if ($msg) $sent_any = true;
    }
    
    // پیام اطلاع حذف خودکار و دکمه دریافت مجدد (بعد از ارسال فایل‌ها)
    // Only show auto-delete message if forward lock is NOT enabled
    if ($sent_any && $file_delete_after > 0 && !$forward_lock) {
        $minutes = floor($file_delete_after / 60);
        $seconds = $file_delete_after % 60;
        $time_display = $minutes > 0 ? "$minutes دقیقه و $seconds ثانیه" : "$seconds ثانیه";
        
        $delete_message = get_auto_delete_message();
        $delete_message = str_replace('{TIME}', $time_display, $delete_message);
        send_message($chat_id, $delete_message);
    }
    
    // پیام دریافت مجدد (جدا از پیام حذف خودکار)
    if ($sent_any && $resend_enabled) {
        $resend_link = "https://t.me/" . BOT_USERNAME . "?start=" . $upload_id;
        $resend_text = "📬 <b>دریافت مجدد رسانه</b>\n\n";
        $resend_text .= "✅ فایل را در پی‌وی دوستان خود یا در پیام‌های ذخیره شده ارسال و بعد دانلود کنید.";
        
        $resend_markup = [
            'inline_keyboard' => [
                [['text' => '📬 دریافت مجدد رسانه', 'url' => $resend_link]]
            ]
        ];
        
        send_message($chat_id, $resend_text, $resend_markup);
    }
    
    return $sent_any;
}

// تنظیمات لایک
function get_like_enabled() {
    $file = __DIR__ . '/like_settings.json';
    if (!file_exists($file)) {
        $default = false;
        file_put_contents($file, json_encode(['enabled' => $default], JSON_PRETTY_PRINT));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return (bool)($data['enabled'] ?? false);
}

function save_like_enabled($enabled) {
    $file = __DIR__ . '/like_settings.json';
    return file_put_contents($file, json_encode(['enabled' => (bool)$enabled], JSON_PRETTY_PRINT));
}

// تنظیمات نمایش تعداد سین
function get_view_count_enabled() {
    $file = __DIR__ . '/view_count_settings.json';
    if (!file_exists($file)) {
        $default = false;
        file_put_contents($file, json_encode(['enabled' => $default], JSON_PRETTY_PRINT));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return (bool)($data['enabled'] ?? false);
}

function save_view_count_enabled($enabled) {
    $file = __DIR__ . '/view_count_settings.json';
    return file_put_contents($file, json_encode(['enabled' => (bool)$enabled], JSON_PRETTY_PRINT));
}

// تنظیمات دریافت مجدد
function get_resend_enabled() {
    $file = __DIR__ . '/resend_settings.json';
    if (!file_exists($file)) {
        $default = false;
        file_put_contents($file, json_encode(['enabled' => $default], JSON_PRETTY_PRINT));
        return $default;
    }
    $data = json_decode(file_get_contents($file), true);
    return (bool)($data['enabled'] ?? false);
}

function save_resend_enabled($enabled) {
    $file = __DIR__ . '/resend_settings.json';
    return file_put_contents($file, json_encode(['enabled' => (bool)$enabled], JSON_PRETTY_PRINT));
}

// مدیریت لایک‌ها
function load_likes_data() {
    $file = __DIR__ . '/likes_data.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['uploads' => []], JSON_PRETTY_PRINT));
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['uploads'] ?? [];
}

function save_likes_data($data) {
    $file = __DIR__ . '/likes_data.json';
    return file_put_contents($file, json_encode(['uploads' => $data], JSON_PRETTY_PRINT));
}

function add_like($upload_id, $user_id, $type = 'like') {
    $data = load_likes_data();
    if (!isset($data[$upload_id])) {
        $data[$upload_id] = ['likes' => [], 'dislikes' => []];
    }
    
    // حذف لایک/دیسلایک قبلی کاربر
    $data[$upload_id]['likes'] = array_values(array_filter($data[$upload_id]['likes'], function($uid) use ($user_id) {
        return $uid != $user_id;
    }));
    $data[$upload_id]['dislikes'] = array_values(array_filter($data[$upload_id]['dislikes'], function($uid) use ($user_id) {
        return $uid != $user_id;
    }));
    
    // اضافه کردن لایک/دیسلایک جدید
    if ($type === 'like') {
        $data[$upload_id]['likes'][] = $user_id;
    } else {
        $data[$upload_id]['dislikes'][] = $user_id;
    }
    
    save_likes_data($data);
    return [
        'likes' => count($data[$upload_id]['likes']),
        'dislikes' => count($data[$upload_id]['dislikes'])
    ];
}

function get_likes($upload_id) {
    $data = load_likes_data();
    if (!isset($data[$upload_id])) {
        return ['likes' => 0, 'dislikes' => 0];
    }
    return [
        'likes' => count($data[$upload_id]['likes'] ?? []),
        'dislikes' => count($data[$upload_id]['dislikes'] ?? [])
    ];
}

// مدیریت بازدیدها
function load_views_data() {
    $file = __DIR__ . '/views_data.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['uploads' => []], JSON_PRETTY_PRINT));
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['uploads'] ?? [];
}

function save_views_data($data) {
    $file = __DIR__ . '/views_data.json';
    return file_put_contents($file, json_encode(['uploads' => $data], JSON_PRETTY_PRINT));
}

function add_view($upload_id, $user_id) {
    $data = load_views_data();
    if (!isset($data[$upload_id])) {
        $data[$upload_id] = [];
    }
    
    // اگر کاربر قبلاً بازدید نکرده، اضافه کن
    if (!in_array($user_id, $data[$upload_id])) {
        $data[$upload_id][] = $user_id;
        save_views_data($data);
    }
    
    return count($data[$upload_id]);
}

function get_views($upload_id) {
    $data = load_views_data();
    return isset($data[$upload_id]) ? count($data[$upload_id]) : 0;
}



// ========== سیستم کیف پول و پرداخت ==========

function get_user_wallet($user_id) {
    $file = __DIR__ . '/wallets.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['users' => []], JSON_PRETTY_PRINT));
    }
    $data = json_decode(file_get_contents($file), true);
    
    if (!isset($data['users'][$user_id])) {
        $data['users'][$user_id] = [
            'balance' => 0,
            'transactions' => []
        ];
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    return $data['users'][$user_id];
}

function add_wallet_balance($user_id, $amount, $description = 'شارژ حساب') {
    $file = __DIR__ . '/wallets.json';
    $data = json_decode(file_get_contents($file), true);
    
    if (!isset($data['users'][$user_id])) {
        $data['users'][$user_id] = [
            'balance' => 0,
            'transactions' => []
        ];
    }
    
    $data['users'][$user_id]['balance'] += $amount;
    $data['users'][$user_id]['transactions'][] = [
        'amount' => $amount,
        'type' => 'credit',
        'description' => $description,
        'date' => date('Y-m-d H:i:s'),
        'jalali_date' => get_jalali_date()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    return $data['users'][$user_id]['balance'];
}

function deduct_wallet_balance($user_id, $amount, $description = 'خرید فایل') {
    $file = __DIR__ . '/wallets.json';
    $data = json_decode(file_get_contents($file), true);
    
    if (!isset($data['users'][$user_id]) || $data['users'][$user_id]['balance'] < $amount) {
        return false;
    }
    
    $data['users'][$user_id]['balance'] -= $amount;
    $data['users'][$user_id]['transactions'][] = [
        'amount' => -$amount,
        'type' => 'debit',
        'description' => $description,
        'date' => date('Y-m-d H:i:s'),
        'jalali_date' => get_jalali_date()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    return true;
}

// تنظیم قیمت برای آپلود
function set_upload_price($upload_id, $price) {
    $uploads = load_uploads();
    if (isset($uploads[$upload_id])) {
        $uploads[$upload_id]['price'] = intval($price);
        $uploads[$upload_id]['is_paid'] = $price > 0;
        save_uploads($uploads);
        return true;
    }
    return false;
}

function get_upload_price($upload_id) {
    $upload = get_upload($upload_id);
    return $upload['price'] ?? 0;
}

function is_upload_paid($upload_id) {
    $upload = get_upload($upload_id);
    return ($upload['is_paid'] ?? false) && ($upload['price'] ?? 0) > 0;
}

// بررسی دسترسی کاربر به آپلود پولی
function has_user_purchased($user_id, $upload_id) {
    $file = __DIR__ . '/purchases.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['purchases' => []], JSON_PRETTY_PRINT));
        return false;
    }
    
    $data = json_decode(file_get_contents($file), true);
    $key = $user_id . ':' . $upload_id;
    return isset($data['purchases'][$key]);
}

function register_purchase($user_id, $upload_id, $amount) {
    $file = __DIR__ . '/purchases.json';
    if (!file_exists($file)) {
        $data = ['purchases' => []];
    } else {
        $data = json_decode(file_get_contents($file), true);
    }
    
    $key = $user_id . ':' . $upload_id;
    $data['purchases'][$key] = [
        'user_id' => $user_id,
        'upload_id' => $upload_id,
        'amount' => $amount,
        'date' => date('Y-m-d H:i:s'),
        'jalali_date' => get_jalali_date()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// تنظیمات پرداخت
function get_payment_settings() {
    $file = __DIR__ . '/payment_settings.json';
    if (!file_exists($file)) {
        $default = [
            'card_number' => '6037-9971-XXXX-XXXX',
            'card_holder' => 'نام صاحب کارت',
            'bank_name' => 'بانک ملی'
        ];
        file_put_contents($file, json_encode($default, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default;
    }
    return json_decode(file_get_contents($file), true);
}

function save_payment_settings($settings) {
    $file = __DIR__ . '/payment_settings.json';
    return file_put_contents($file, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// وضعیت ربات (روشن/خاموش)
function is_bot_enabled() {
    $file = __DIR__ . '/bot_status.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['enabled' => true], JSON_PRETTY_PRINT));
        return true;
    }
    $data = json_decode(file_get_contents($file), true);
    return (bool)($data['enabled'] ?? true);
}

function set_bot_enabled($enabled) {
    $file = __DIR__ . '/bot_status.json';
    return file_put_contents($file, json_encode(['enabled' => (bool)$enabled], JSON_PRETTY_PRINT));
}

// ذخیره رسید پرداخت
function save_payment_receipt($user_id, $upload_id, $amount, $receipt_file_id) {
    $file = __DIR__ . '/payment_receipts.json';
    if (!file_exists($file)) {
        $data = ['receipts' => []];
    } else {
        $data = json_decode(file_get_contents($file), true);
    }
    
    $receipt_id = 'rcpt_' . bin2hex(random_bytes(6));
    $data['receipts'][$receipt_id] = [
        'id' => $receipt_id,
        'user_id' => $user_id,
        'upload_id' => $upload_id,
        'amount' => $amount,
        'receipt_file_id' => $receipt_file_id,
        'status' => 'pending',
        'date' => date('Y-m-d H:i:s'),
        'jalali_date' => get_jalali_date()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    return $receipt_id;
}

function get_pending_receipts() {
    $file = __DIR__ . '/payment_receipts.json';
    if (!file_exists($file)) {
        return [];
    }
    
    $data = json_decode(file_get_contents($file), true);
    $pending = [];
    
    foreach ($data['receipts'] as $receipt_id => $receipt) {
        if ($receipt['status'] === 'pending') {
            $pending[$receipt_id] = $receipt;
        }
    }
    
    return $pending;
}

function approve_receipt($receipt_id) {
    $file = __DIR__ . '/payment_receipts.json';
    $data = json_decode(file_get_contents($file), true);
    
    if (isset($data['receipts'][$receipt_id])) {
        $receipt = $data['receipts'][$receipt_id];
        
        // بررسی که قبلاً تایید نشده باشد
        if ($receipt['status'] !== 'pending') {
            return null; // قبلاً پردازش شده
        }
        
        $data['receipts'][$receipt_id]['status'] = 'approved';
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        // شارژ کیف پول کاربر
        add_wallet_balance($receipt['user_id'], $receipt['amount'], 'تایید رسید پرداخت');
        
        return $receipt;
    }
    
    return null;
}

function reject_receipt($receipt_id) {
    $file = __DIR__ . '/payment_receipts.json';
    $data = json_decode(file_get_contents($file), true);
    
    if (isset($data['receipts'][$receipt_id])) {
        $receipt = $data['receipts'][$receipt_id];
        
        // بررسی که قبلاً رد نشده باشد
        if ($receipt['status'] !== 'pending') {
            return null; // قبلاً پردازش شده
        }
        
        $data['receipts'][$receipt_id]['status'] = 'rejected';
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        return $receipt;
    }
    
    return null;
}


// ========== تبدیل تاریخ به شمسی ==========

function gregorian_to_jalali($g_y, $g_m, $g_d) {
    $g_days_in_month = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    $j_days_in_month = [31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29];
    
    $gy = $g_y - 1600;
    $gm = $g_m - 1;
    $gd = $g_d - 1;
    
    $g_day_no = 365 * $gy + floor(($gy + 3) / 4) - floor(($gy + 99) / 100) + floor(($gy + 399) / 400);
    
    for ($i = 0; $i < $gm; ++$i)
        $g_day_no += $g_days_in_month[$i];
    if ($gm > 1 && (($gy % 4 == 0 && $gy % 100 != 0) || ($gy % 400 == 0)))
        $g_day_no++;
    $g_day_no += $gd;
    
    $j_day_no = $g_day_no - 79;
    
    $j_np = floor($j_day_no / 12053);
    $j_day_no = $j_day_no % 12053;
    
    $jy = 979 + 33 * $j_np + 4 * floor($j_day_no / 1461);
    
    $j_day_no %= 1461;
    
    if ($j_day_no >= 366) {
        $jy += floor(($j_day_no - 1) / 365);
        $j_day_no = ($j_day_no - 1) % 365;
    }
    
    $j_month = 0;
    for ($i = 0; $i < 11 && $j_day_no >= $j_days_in_month[$i]; ++$i) {
        $j_day_no -= $j_days_in_month[$i];
        $j_month++;
    }
    $jd = $j_day_no + 1;
    $jm = $j_month + 1;
    
    return [$jy, $jm, $jd];
}

function get_jalali_date($timestamp = null) {
    if ($timestamp === null) {
        $timestamp = time();
    }
    
    $g_y = date('Y', $timestamp);
    $g_m = date('m', $timestamp);
    $g_d = date('d', $timestamp);
    
    list($j_y, $j_m, $j_d) = gregorian_to_jalali($g_y, $g_m, $g_d);
    
    $j_months = [
        1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد',
        4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
        7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
        10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
    ];
    
    $time = date('H:i', $timestamp);
    
    return $j_d . ' ' . $j_months[$j_m] . ' ' . $j_y . ' - ' . $time;
}

function get_jalali_date_short($timestamp = null) {
    if ($timestamp === null) {
        $timestamp = time();
    }
    
    $g_y = date('Y', $timestamp);
    $g_m = date('m', $timestamp);
    $g_d = date('d', $timestamp);
    
    list($j_y, $j_m, $j_d) = gregorian_to_jalali($g_y, $g_m, $g_d);
    
    return $j_y . '/' . str_pad($j_m, 2, '0', STR_PAD_LEFT) . '/' . str_pad($j_d, 2, '0', STR_PAD_LEFT);
}

// ========== آمار فروش ==========

function get_sales_stats() {
    $uploads = load_uploads();
    $purchases_file = __DIR__ . '/purchases.json';
    
    $total_uploads = count($uploads);
    $paid_uploads = 0;
    $free_uploads = 0;
    $total_revenue = 0;
    $total_sales = 0;
    
    // شمارش آپلودهای پولی و رایگان
    foreach ($uploads as $upload) {
        if (isset($upload['is_paid']) && $upload['is_paid'] === true && ($upload['price'] ?? 0) > 0) {
            $paid_uploads++;
        } else {
            $free_uploads++;
        }
    }
    
    // محاسبه درآمد کل
    if (file_exists($purchases_file)) {
        $purchases_data = json_decode(file_get_contents($purchases_file), true);
        $purchases = $purchases_data['purchases'] ?? [];
        
        foreach ($purchases as $purchase) {
            $total_revenue += $purchase['amount'] ?? 0;
            $total_sales++;
        }
    }
    
    // محاسبه درصد فروش
    $sales_percentage = $paid_uploads > 0 ? round(($total_sales / $paid_uploads) * 100, 2) : 0;
    
    return [
        'total_uploads' => $total_uploads,
        'paid_uploads' => $paid_uploads,
        'free_uploads' => $free_uploads,
        'total_revenue' => $total_revenue,
        'total_sales' => $total_sales,
        'sales_percentage' => $sales_percentage
    ];
}


// ========== سیستم اشتراک ==========

function load_subscriptions() {
    $file = __DIR__ . '/subscriptions.json';
    if (!file_exists($file)) {
        $default = [
            'users' => [],
            'settings' => [
                '1_month' => ['duration_days' => 30, 'price' => 25000, 'name' => 'یک ماهه'],
                '2_month' => ['duration_days' => 60, 'price' => 45000, 'name' => 'دو ماهه'],
                '3_month' => ['duration_days' => 90, 'price' => 70000, 'name' => 'سه ماهه']
            ]
        ];
        file_put_contents($file, json_encode($default, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default;
    }
    return json_decode(file_get_contents($file), true);
}

function save_subscriptions($data) {
    $file = __DIR__ . '/subscriptions.json';
    return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function get_subscription_settings() {
    $data = load_subscriptions();
    return $data['settings'] ?? [];
}

function save_subscription_settings($settings) {
    $data = load_subscriptions();
    $data['settings'] = $settings;
    return save_subscriptions($data);
}

function get_user_subscription($user_id) {
    $data = load_subscriptions();
    if (isset($data['users'][$user_id])) {
        $sub = $data['users'][$user_id];
        // بررسی انقضا
        if (strtotime($sub['expires_at']) > time()) {
            return $sub;
        }
    }
    return null;
}

function has_active_subscription($user_id) {
    $sub = get_user_subscription($user_id);
    return $sub !== null;
}

function purchase_subscription($user_id, $plan_type) {
    $settings = get_subscription_settings();
    
    if (!isset($settings[$plan_type])) {
        return false;
    }
    
    $plan = $settings[$plan_type];
    $data = load_subscriptions();
    
    // محاسبه تاریخ انقضا
    $start_date = time();
    $expires_at = date('Y-m-d H:i:s', strtotime('+' . $plan['duration_days'] . ' days', $start_date));
    $jalali_expires = get_jalali_date(strtotime($expires_at));
    
    $data['users'][$user_id] = [
        'plan_type' => $plan_type,
        'plan_name' => $plan['name'],
        'price' => $plan['price'],
        'duration_days' => $plan['duration_days'],
        'started_at' => date('Y-m-d H:i:s'),
        'expires_at' => $expires_at,
        'jalali_started' => get_jalali_date(),
        'jalali_expires' => $jalali_expires,
        'downloads_count' => 0
    ];
    
    save_subscriptions($data);
    
    // ثبت در تراکنش‌های کیف پول
    deduct_wallet_balance($user_id, $plan['price'], 'خرید اشتراک ' . $plan['name']);
    
    return true;
}

function increment_subscription_downloads($user_id) {
    $data = load_subscriptions();
    if (isset($data['users'][$user_id])) {
        $data['users'][$user_id]['downloads_count']++;
        save_subscriptions($data);
    }
}

function get_subscription_stats($user_id) {
    $sub = get_user_subscription($user_id);
    if (!$sub) {
        return null;
    }
    
    $remaining_days = ceil((strtotime($sub['expires_at']) - time()) / 86400);
    
    return [
        'plan_name' => $sub['plan_name'],
        'expires_at' => $sub['jalali_expires'] ?? $sub['expires_at'],
        'remaining_days' => $remaining_days,
        'downloads_count' => $sub['downloads_count'] ?? 0
    ];
}
