<?php
/**
 * وب‌هوک تلگرام برای دریافت و پردازش پیام‌های ورودی
 * 
 * این فایل پیام‌های دریافتی از تلگرام را پردازش می‌کند
 */

// لود کردن فایل‌های مورد نیاز
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/admin_actions.php';
require_once __DIR__ . '/admin.php';

// دریافت داده‌های ورودی از تلگرام
$update = json_decode(file_get_contents('php://input'), true);

// ثبت لاگ برای دیباگ
file_put_contents('update_log.txt', date('Y-m-d H:i:s') . ' - ' . json_encode($update, JSON_UNESCAPED_UNICODE) . PHP_EOL, FILE_APPEND);

// پردازش پیام‌های دریافتی
if (isset($update['message'])) {
    processMessage($update['message']);
} elseif (isset($update['callback_query'])) {
    processCallbackQuery($update['callback_query']);
} else {
    error_log("Unknown update type: " . json_encode($update, JSON_UNESCAPED_UNICODE));
}

/**
 * پردازش پیام‌های متنی دریافتی
 * 
 * @param array $message پیام دریافتی
 */
function processMessage($message) {
    // دریافت اطلاعات پیام
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $text = $message['text'] ?? '';
    $username = $message['from']['username'] ?? '';
    $firstName = $message['from']['first_name'] ?? '';
    $lastName = $message['from']['last_name'] ?? '';
    
    // ذخیره یا به‌روزرسانی اطلاعات کاربر
    $user = [
        'id' => $userId,
        'username' => $username,
        'first_name' => $firstName,
        'last_name' => $lastName
    ];
    saveUser($user);
    
    // بررسی دستور شروع
    if ($text === '/start') {
        // دریافت آمار برای نمایش
        $stats = getUserStats();
        
        // دریافت تنظیمات ربات از فایل JSON
        $settings = getBotSettingsFromFile();
        $startMessage = $settings['start_message'] ?? "👋 سلام <b>{first_name}</b>!\n\nبه ربات دانلود اینستاگرام خوش آمدید. 🚀";
        
        // جایگزینی متغیرها در متن
        $welcomeMessage = str_replace(
            ['{first_name}', '{total_users}', '{total_downloads}'],
            [$firstName, number_format($stats['total_users']), number_format($stats['total_downloads'])],
            $startMessage
        );
        
        // ارسال پیام با کیبورد معمولی
        sendMessage($chatId, $welcomeMessage, getMainKeyboard());
        
        // اگر دکمه‌های URL داریم، اونها رو هم ارسال کن
        $urlKeyboard = getUrlButtonsKeyboard();
        if ($urlKeyboard) {
            sendMessage($chatId, "🔗 <b>لینک‌های مفید:</b>", $urlKeyboard);
        }
        
        return;
    }
    
    // بررسی دستورات لغو و پنل (اولویت بالا)
    if ($text === '/panel' || $text === '/cancel' || $text === 'لغو') {
        // پاک کردن session اگر وجود دارد
        if (isAdmin($userId)) {
            clearUserSession($userId);
            
            if ($text === '/panel') {
                handleAdminPanel($chatId);
            } else {
                sendMessage($chatId, "✅ عملیات لغو شد.", getMainKeyboard());
            }
        } else {
            // برای کاربران عادی
            if ($text === '/panel') {
            $errorMessage = "⛔ <b>عدم دسترسی</b>\n\n";
            $errorMessage .= "شما به این بخش دسترسی ندارید.";
            sendMessage($chatId, $errorMessage);
            } else {
                sendMessage($chatId, "✅ عملیات لغو شد.", getMainKeyboard());
            }
        }
        return;
    }
    
    // بررسی session کاربر برای اکشن‌های ادمین
    if (isAdmin($userId) && handleUserSessionMessage($message)) {
        return;
    }
    
    // بررسی عضویت کاربر در کانال‌های اجباری
    $channelCheck = checkRequiredChannels($userId);
    if (!$channelCheck['joined']) {
        sendForceJoinMessage($userId, $channelCheck['not_joined']);
        return;
    }
    
    // بررسی محدودیت نرخ درخواست
    if (checkRateLimit($userId)) {
        $limitMessage = "⚠️ <b>محدودیت درخواست</b>\n\n";
        $limitMessage .= "شما بیش از حد مجاز درخواست ارسال کرده‌اید. لطفاً کمی صبر کنید و دوباره تلاش کنید.";
        sendMessage($chatId, $limitMessage);
        return;
    }
    
    // پردازش دکمه‌های اصلی
    if ($text === '📥 دانلود پست/ریل') {
        $postMessage = "📥 <b>دانلود پست یا ریل اینستاگرام</b>\n\n";
        $postMessage .= "لطفاً لینک پست یا ریل را ارسال کنید:\n\n";
        $postMessage .= "مثال:\n";
        $postMessage .= "• https://www.instagram.com/p/ABC123/\n";
        $postMessage .= "• https://www.instagram.com/reel/XYZ789/";
        sendMessage($chatId, $postMessage);
        return;
    } elseif ($text === '📹 دانلود استوری') {
        $storyMessage = "📹 <b>دانلود استوری اینستاگرام</b>\n\n";
        $storyMessage .= "<b>✅ روش 1: لینک مستقیم (پیشنهادی)</b>\n";
        $storyMessage .= "• روی استوری کلیک کنید\n";
        $storyMessage .= "• گزینه Share → Copy Link را بزنید\n";
        $storyMessage .= "• لینک را اینجا ارسال کنید\n\n";
        $storyMessage .= "<b>📝 روش 2: نام کاربری</b>\n";
        $storyMessage .= "• نام کاربری را بفرستید (بدون @)\n";
        $storyMessage .= "• مثال: havale.yab\n\n";
        $storyMessage .= "💡 <b>انواع قابل دانلود:</b>\n";
        $storyMessage .= "• استوری‌های معمولی\n";
        $storyMessage .= "• هایلایت‌ها (Highlights)\n\n";
        $storyMessage .= "⚠️ نکته: استوری‌های حساب‌های خصوصی قابل دانلود نیستند.";
        sendMessage($chatId, $storyMessage);
        return;
    } elseif ($text === '❓ راهنما') {
        $helpMessage = "📚 <b>راهنمای استفاده از ربات</b>\n\n";
        $helpMessage .= "🔹 برای دانلود پست یا ریل:\n";
        $helpMessage .= "لینک پست یا ریل را ارسال کنید\n\n";
        $helpMessage .= "🔹 برای دانلود استوری:\n";
        $helpMessage .= "لینک استوری یا نام کاربری را ارسال کنید\n\n";
        $helpMessage .= "🔹 فرمت‌های پشتیبانی شده:\n";
        $helpMessage .= "• پست: /p/...\n";
        $helpMessage .= "• ریل: /reel/...\n";
        $helpMessage .= "• استوری: لینک مستقیم یا نام کاربری\n";
        $helpMessage .= "• هایلایت: /s/...\n\n";
        $helpMessage .= "💡 فقط کافیست لینک را ارسال کنید!";
        sendMessage($chatId, $helpMessage);
        return;
    } elseif ($text === '📊 درباره ربات') {
        // دریافت آمار
        $stats = getUserStats();
        
        $aboutMessage = "🤖 <b>درباره ربات دانلود اینستاگرام</b>\n\n";
        $aboutMessage .= "این ربات با استفاده از قدرتمندترین API های موجود، محتوای اینستاگرام را برای شما دانلود می‌کند.\n\n";
        
        $aboutMessage .= "✨ <b>قابلیت‌ها:</b>\n";
        $aboutMessage .= "✅ دانلود پست (عکس و ویدیو)\n";
        $aboutMessage .= "✅ دانلود ریل\n";
        $aboutMessage .= "✅ دانلود استوری\n";
        $aboutMessage .= "✅ دانلود هایلایت\n";
        $aboutMessage .= "✅ دانلود آلبوم (چند رسانه‌ای)\n";
        $aboutMessage .= "✅ سرعت بالا و کیفیت HD\n";
        $aboutMessage .= "✅ رایگان و بدون محدودیت\n\n";
        
        $aboutMessage .= "📊 <b>آمار ربات:</b>\n";
        $aboutMessage .= "👥 کاربران فعال: <b>" . number_format($stats['total_users']) . "</b>\n";
        $aboutMessage .= "📥 کل دانلودها: <b>" . number_format($stats['total_downloads']) . "</b>\n";
        $aboutMessage .= "🔥 کاربران فعال امروز: <b>" . number_format($stats['active_today']) . "</b>\n";
        $aboutMessage .= "📈 دانلودهای امروز: <b>" . number_format($stats['downloads_today']) . "</b>\n\n";
        
        $aboutMessage .= "🛡 <b>امنیت و حریم خصوصی:</b>\n";
        $aboutMessage .= "• هیچ اطلاعاتی ذخیره نمی‌شود\n";
        $aboutMessage .= "• دانلود مستقیم و ایمن\n";
        $aboutMessage .= "• بدون نیاز به لاگین\n\n";
        
        $aboutMessage .= "💎 <b>نسخه:</b> 2.0\n";
        $aboutMessage .= "📅 <b>آخرین بروزرسانی:</b> " . date('Y/m/d') . "\n\n";
        
        $aboutMessage .= "💬 برای سفارش ربات یا خرید VPN، از دکمه پایین استفاده کنید.";
        
        sendMessage($chatId, $aboutMessage);
        return;
    } elseif ($text === 'سفارش ربات | خرید VPN') {
        $supportMessage = "🔥 می‌خوای یه ربات حرفه‌ای مخصوص خودت داشته باشی که همه کارات رو اتومات کنه؟\n";
        $supportMessage .= "🌐 یا دنبال یه VPN پرسرعت و بدون قطعی برای دور زدن محدودیتایی؟\n\n";
        $supportMessage .= "فرصت رو از دست نده! همین الان می‌تونی ربات اختصاصی خودتو سفارش بدی یا یه VPN قدرتمند تهیه کنی و راحت نفس بکشی 😎\n\n";
        $supportMessage .= "📩 برای سفارش و دریافت اطلاعات بیشتر پیام بده👇";
        
        $inlineKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '📩 ارتباط با پشتیبانی', 'url' => 'https://t.me/amirth1998']
                ]
            ]
        ];
        
        sendMessage($chatId, $supportMessage, $inlineKeyboard);
        return;
    }
    
    // پردازش لینک‌های اینستاگرام
    if (strpos($text, 'instagram.com') !== false) {
        processInstagramLink($chatId, $userId, $text);
        
        // ثبت آمار استفاده
        handleStats($userId, 'download', 'link');
        return;
    }
    
    // پردازش نام کاربری اینستاگرام (برای استوری)
    if (preg_match('/^@?([a-zA-Z0-9._]{1,30})$/', $text, $matches)) {
        $username = $matches[1];
        
        // دریافت استوری
        $storyMessage = "📹 <b>دانلود استوری</b>\n\n";
        $storyMessage .= "در حال دریافت استوری‌های @{$username}...";
        sendMessage($chatId, $storyMessage);
        
        // استراتژی 1: اول با Instagram120 امتحان می‌کنیم
        error_log("Strategy 1: Trying Instagram120 for stories: $username");
        $result = getStoriesFromInstagram120($username);
        
        // لاگ کامل پاسخ
        error_log("Story API Response: " . json_encode($result, JSON_UNESCAPED_UNICODE));
        
        if ($result['success']) {
            $data = $result['data'];
            error_log("Story Data: " . json_encode($data, JSON_UNESCAPED_UNICODE));
            
            $stories = [];
            $items = [];
            
            // فرمت Instagram120: {status: "success", data: {items: [...]}}
            if (isset($data['status']) && $data['status'] === 'success' && isset($data['data']['items']) && is_array($data['data']['items'])) {
                $items = $data['data']['items'];
                error_log("Detected Instagram120 story format");
            }
            // فرمت RapidAPI: {data: {items: [...]}}
            elseif (isset($data['data']) && isset($data['data']['items']) && is_array($data['data']['items'])) {
                $items = $data['data']['items'];
                error_log("Detected RapidAPI story format");
            }
            
            if (!empty($items)) {
                sendMessage($chatId, "⏳ در حال دانلود و ارسال استوری @{$username}...");
                
                foreach ($items as $index => $item) {
                    error_log("Processing story " . ($index + 1) . ": " . json_encode($item, JSON_UNESCAPED_UNICODE));
                    
                    // کپشن ساده
                    $storyCaption = "📥 " . BOT_USERNAME;
                    
                    $sent = false;
                    
                    // پشتیبانی از فرمت‌های مختلف
                    // Instagram120 format: video_url یا display_url
                    if (isset($item['video_url']) && !empty($item['video_url'])) {
                        sendVideo($chatId, $item['video_url'], $storyCaption);
                        $sent = true;
                    } elseif (isset($item['display_url']) && !empty($item['display_url'])) {
                        sendPhoto($chatId, $item['display_url'], $storyCaption);
                        $sent = true;
                    }
                    // RapidAPI format: video_versions یا image_versions
                    elseif (isset($item['video_versions']) && !empty($item['video_versions'])) {
                        $videoUrl = $item['video_versions'][0]['url'];
                        sendVideo($chatId, $videoUrl, $storyCaption);
                        $sent = true;
                    } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
                        $imageUrl = $item['image_versions2']['candidates'][0]['url'];
                        sendPhoto($chatId, $imageUrl, $storyCaption);
                        $sent = true;
                    }
                    
                    if (!$sent) {
                        error_log("Story not sent - no valid media URL found");
                    }
                    
                    usleep(500000); // 500ms تاخیر
                }
            } else {
                // پیام بهتر برای کاربر
                $noStoryMsg = "⚠️ <b>استوری در دسترس نیست</b>\n\n";
                $noStoryMsg .= "دلایل احتمالی:\n";
                $noStoryMsg .= "• حساب خصوصی است\n";
                $noStoryMsg .= "• استوری فعالی ندارد\n";
                $noStoryMsg .= "• API نمی‌تواند دسترسی داشته باشد\n\n";
                $noStoryMsg .= "💡 <b>راه حل:</b>\n";
                $noStoryMsg .= "لینک مستقیم استوری را ارسال کنید:\n";
                $noStoryMsg .= "1. روی استوری کلیک کنید\n";
                $noStoryMsg .= "2. گزینه Share → Copy Link\n";
                $noStoryMsg .= "3. لینک را اینجا بفرستید";
                
                sendMessage($chatId, $noStoryMsg);
            }
        } else {
            // استراتژی 2: اگر Instagram120 موفق نبود، از RapidAPI قدیمی امتحان می‌کنیم
            if (RAPIDAPI_KEY !== 'YOUR_RAPIDAPI_KEY_HERE') {
                error_log("Strategy 2: Instagram120 failed, trying RapidAPI for stories: $username");
                $result = getStoriesFromRapidAPI($username);
                
                if ($result['success'] && isset($result['data']['data']['items']) && !empty($result['data']['data']['items'])) {
                    $items = $result['data']['data']['items'];
                    sendMessage($chatId, "⏳ در حال دانلود و ارسال استوری @{$username}...");
                    
                    foreach ($items as $index => $item) {
                        // کپشن ساده
                        $storyCaption = "📥 " . BOT_USERNAME;
                        
                        if (isset($item['video_versions']) && !empty($item['video_versions'])) {
                            sendVideo($chatId, $item['video_versions'][0]['url'], $storyCaption);
                        } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
                            sendPhoto($chatId, $item['image_versions2']['candidates'][0]['url'], $storyCaption);
                        }
                        
                        usleep(500000);
                    }
                    
                    // ثبت آمار استفاده
                    handleStats($userId, 'download', 'story');
                    return;
                }
            }
            
            // همه API ها موفق نبودند
            sendMessage($chatId, "❌ خطا در دریافت استوری. لطفاً بعداً تلاش کنید.");
        }
        
        // ثبت آمار استفاده
        handleStats($userId, 'download', 'story');
        return;
    }
    
    // بررسی دکمه‌های سفارشی
    $settings = getBotSettingsFromFile();
    $customButtonFound = false;
    
    if (!empty($settings['buttons'])) {
        foreach ($settings['buttons'] as $row) {
            foreach ($row as $button) {
                if (isset($button['text']) && $button['text'] === $text) {
                    // دکمه سفارشی پیدا شد
                    if (isset($button['type']) && $button['type'] === 'text' && isset($button['response'])) {
                        // دکمه با متن پاسخ
                        sendMessage($chatId, $button['response']);
                        $customButtonFound = true;
                        
                        // ثبت آمار
                        handleStats($userId, 'custom_button', $text);
                        return;
                    }
                }
            }
        }
    }
    
    // اگر دکمه سفارشی بود که پیدا شد، دیگر ادامه نده
    if ($customButtonFound) {
        return;
    }
    
    // پیام راهنما برای ورودی‌های نامعتبر
    $helpMessage = "❓ <b>راهنما</b>\n\n";
    $helpMessage .= "لطفاً یکی از موارد زیر را ارسال کنید:\n\n";
    $helpMessage .= "📥 <b>برای دانلود پست یا ریل:</b>\n";
    $helpMessage .= "• لینک کامل پست یا ریل\n";
    $helpMessage .= "مثال: https://www.instagram.com/p/ABC123/\n\n";
    $helpMessage .= "📹 <b>برای دانلود استوری:</b>\n";
    $helpMessage .= "• نام کاربری (بدون @)\n";
    $helpMessage .= "مثال: havale.yab\n\n";
    $helpMessage .= "یا از دکمه‌های زیر استفاده کنید 👇";
    
    sendMessage($chatId, $helpMessage, getMainKeyboard());
}

/**
 * پردازش کالبک‌های دریافتی از دکمه‌های اینلاین
 * 
 * @param array $callbackQuery داده‌های کالبک
 */
function processCallbackQuery($callbackQuery) {
    $chatId = $callbackQuery['message']['chat']['id'];
    $userId = $callbackQuery['from']['id'];
    $messageId = $callbackQuery['message']['message_id'];
    $data = $callbackQuery['data'];
    
    // بررسی دکمه بررسی عضویت
    if ($data === 'check_join') {
        $channelCheck = checkRequiredChannels($userId);
        
        if ($channelCheck['joined']) {
            // کاربر در همه کانال‌ها عضو است
            $successMessage = "✅ <b>عضویت تأیید شد</b>\n\n";
            $successMessage .= "با تشکر از عضویت شما در کانال(های) ما.\n";
            $successMessage .= "اکنون می‌توانید از تمامی امکانات ربات استفاده کنید.";
            
            sendMessage($chatId, $successMessage, getMainKeyboard());
        } else {
            // کاربر هنوز در همه کانال‌ها عضو نیست
            sendForceJoinMessage($userId, $channelCheck['not_joined']);
        }
        
        // پاسخ به کالبک
        answerCallbackQuery($callbackQuery['id'], "وضعیت عضویت بررسی شد.");
        return;
    }
    
    // پردازش سایر کالبک‌ها
    if (strpos($data, 'download_') === 0) {
        // دانلود مدیا
        $parts = explode('_', $data);
        $type = $parts[1] ?? '';
        $url = $parts[2] ?? '';
        
        if ($type && $url) {
            // دانلود و ارسال مدیا
            downloadAndSendMedia($chatId, $userId, $type, $url);
            
            // ثبت آمار استفاده
            handleStats($userId, 'download', $type);
        }
        
        // پاسخ به کالبک
        answerCallbackQuery($callbackQuery['id'], "درخواست دانلود دریافت شد.");
        return;
    }
    
    // پردازش دکمه‌های پنل مدیریت
    if (strpos($data, 'admin_') === 0 || strpos($data, 'deladmin_') === 0 || strpos($data, 'delchannel_') === 0 || strpos($data, 'delbutton_') === 0) {
        handleAdminAction($callbackQuery);
        return;
    }
    
    // پاسخ پیش‌فرض به کالبک
    answerCallbackQuery($callbackQuery['id'], "عملیات انجام شد.");
}

/**
 * پردازش لینک‌های اینستاگرام
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 * @param string $link لینک اینستاگرام
 */
function processInstagramLink($chatId, $userId, $link) {
    // ارسال پیام در حال پردازش
    $processingMessage = "⏳ <b>در حال پردازش لینک...</b>\n\n";
    $processingMessage .= "لطفاً کمی صبر کنید.";
    $sentMessage = sendMessage($chatId, $processingMessage);
    
    // تشخیص نوع لینک
    if (strpos($link, '/p/') !== false || strpos($link, '/reel/') !== false || strpos($link, '/tv/') !== false) {
        // لینک پست، ریل یا IGTV
        
        // استراتژی 1: اول از وب سرویس fast-creat.ir تلاش می‌کنیم
        error_log("Strategy 1: Trying Fast-Creat API for link: $link");
        $result = callInstagramAPI('post2', ['url' => $link]);
        
        if ($result['success']) {
            error_log("Fast-Creat API success for post/reel");
            $sendResult = sendPostContent($chatId, $userId, $result['data']);
            // اگر ارسال موفق بود، return
            if (!isset($sendResult['success']) || $sendResult['success'] !== false) {
                return;
            }
            error_log("Fast-Creat API returned data but sendPostContent failed, trying next strategy");
        }
        
        // استراتژی 2: اگر وب سرویس موفق نبود، از Instagram120 استفاده می‌کنیم
        error_log("Strategy 2: Fast-Creat failed, trying Instagram120 for link: $link");
        $result = getPostFromInstagram120($link);
        
        if ($result['success']) {
            error_log("Instagram120 success for post/reel");
            $sendResult = sendPostContent($chatId, $userId, $result['data']);
            // اگر ارسال موفق بود، return
            if (!isset($sendResult['success']) || $sendResult['success'] !== false) {
                return;
            }
            error_log("Instagram120 returned data but sendPostContent failed, trying next strategy");
        }
        
        // استراتژی 3: اگر Instagram120 هم موفق نبود، RapidAPI قدیمی را امتحان می‌کنیم
        if (RAPIDAPI_KEY !== 'YOUR_RAPIDAPI_KEY_HERE') {
            error_log("Strategy 3: Instagram120 failed, trying RapidAPI for link: $link");
            $result = getPostFromRapidAPI($link);
            
            if ($result['success'] && isset($result['data']['data'])) {
                error_log("RapidAPI success for post/reel");
                $sendResult = sendPostContent($chatId, $userId, $result['data']);
                // اگر ارسال موفق بود، return
                if (!isset($sendResult['success']) || $sendResult['success'] !== false) {
                    return;
                }
                error_log("RapidAPI returned data but sendPostContent failed");
            }
        }
        
        // همه API ها موفق نبودند
        error_log("All APIs failed for link: $link");
        
        $errorMessage = "❌ <b>خطا در دریافت اطلاعات</b>\n\n";
        $errorMessage .= "متأسفانه نتوانستیم این پست را دانلود کنیم.\n\n";
        $errorMessage .= "دلایل احتمالی:\n";
        $errorMessage .= "• پست خصوصی است\n";
        $errorMessage .= "• پست حذف شده است\n";
        $errorMessage .= "• لینک نامعتبر است\n\n";
        $errorMessage .= "💡 لطفاً لینک را بررسی کنید و دوباره تلاش کنید.";
        sendMessage($chatId, $errorMessage);
    } elseif (strpos($link, '/stories/') !== false || strpos($link, '/s/aGlnaGxpZ2h0') !== false) {
        // لینک استوری یا هایلایت
        // فرمت استوری: https://www.instagram.com/stories/USERNAME/...
        // فرمت هایلایت: https://www.instagram.com/s/aGlnaGxpZ2h0...
        
        // استخراج username برای نمایش (فقط برای استوری معمولی)
        $username = '';
        if (preg_match('/instagram\.com\/stories\/([a-zA-Z0-9._]+)/', $link, $matches)) {
            $username = $matches[1];
        }
        
        sendMessage($chatId, "📹 در حال دریافت استوری...");
        
        // استراتژی 1: اول از Fast-Creat API با لینک مستقیم استفاده می‌کنیم
        error_log("Strategy 1: Trying Fast-Creat API for story link: $link");
        $result = getStoryFromFastCreatAPI($link);
        
        if ($result['success'] && isset($result['data'])) {
            error_log("Fast-Creat API success for story");
            
            // پردازش پاسخ Fast-Creat API
            $data = $result['data'];
            
            // بررسی فرمت پاسخ
            // فرمت جدید fast-creat.ir: {ok: true, result: {media: [...]}}
            if (isset($data['ok']) && $data['ok'] === true && isset($data['result']['media']) && is_array($data['result']['media'])) {
                error_log("Detected fast-creat.ir format with media array");
                $items = $data['result']['media'];
            }
            // فرمت ممکن: {status: "successfully", result: {media: [...]}}
            elseif (isset($data['status']) && $data['status'] === 'successfully' && isset($data['result']['media']) && is_array($data['result']['media'])) {
                error_log("Detected successfully format with media array");
                $items = $data['result']['media'];
            }
            // فرمت ممکن: {ok: true, result: {result: [...]}}
            elseif (isset($data['ok']) && $data['ok'] === true && isset($data['result']['result'])) {
                error_log("Detected ok format with result");
                $items = is_array($data['result']['result']) ? $data['result']['result'] : [$data['result']['result']];
            }
            // فرمت ممکن: {status: "successfully", result: {result: [...]}}
            elseif (isset($data['status']) && $data['status'] === 'successfully' && isset($data['result']['result'])) {
                error_log("Detected successfully format with result");
                $items = is_array($data['result']['result']) ? $data['result']['result'] : [$data['result']['result']];
            }
            // فرمت ممکن: {status: "success", data: {...}}
            elseif (isset($data['status']) && $data['status'] === 'success' && isset($data['data'])) {
                error_log("Detected success format with data");
                $items = [$data['data']];
            }
            // فرمت پیش‌فرض
            else {
                error_log("Using default format");
                $items = [$data];
            }
            
            if (!empty($items)) {
                $displayUsername = !empty($username) ? "@{$username}" : "اینستاگرام";
                sendMessage($chatId, "⏳ در حال دانلود و ارسال استوری " . $displayUsername . "...");
                
                foreach ($items as $index => $item) {
                    // لاگ کامل آیتم برای دیباگ
                    error_log("Story item " . ($index + 1) . ": " . json_encode($item, JSON_UNESCAPED_UNICODE));
                    
                    // کپشن ساده بدون شماره
                    $storyCaption = "📥 " . BOT_USERNAME;
                    
                    $sent = false;
                    
                    // پشتیبانی از فرمت‌های مختلف
                    // فرمت fast-creat.ir: {url: "...", type: "video"}
                    if (isset($item['url']) && isset($item['type'])) {
                        // فیلتر: نباید فایل صوتی باشد (sf=.mp3)
                        $isAudioFile = strpos($item['url'], 'sf=.mp3') !== false || strpos($item['url'], '.mp3') !== false;
                        
                        if ($item['type'] === 'video' && !$isAudioFile) {
                            error_log("Sending video (url+type): " . substr($item['url'], 0, 100));
                            
                            // برای URL های خیلی طولانی، ویدیو را دانلود و آپلود می‌کنیم
                            if (strlen($item['url']) > 200) {
                                error_log("URL is too long, downloading and uploading video...");
                                $videoFile = downloadAndUploadVideo($chatId, $item['url'], $storyCaption);
                                $sent = $videoFile;
                            } else {
                                sendVideo($chatId, $item['url'], $storyCaption);
                                $sent = true;
                            }
                        } elseif ($item['type'] === 'image' || $item['type'] === 'photo') {
                            error_log("Sending photo (url+type): " . substr($item['url'], 0, 100));
                            sendPhoto($chatId, $item['url'], $storyCaption);
                            $sent = true;
                        } elseif ($isAudioFile) {
                            error_log("Skipping audio file: " . substr($item['url'], 0, 100));
                            // نادیده می‌گیریم - فایل صوتی را نمی‌خواهیم
                            $sent = true; // به عنوان ارسال شده علامت می‌زنیم تا خطا ندهد
                        }
                    }
                    // فرمت قدیمی: is_video + video_url
                    elseif (isset($item['is_video']) && $item['is_video'] === true && isset($item['video_url'])) {
                        error_log("Sending video (is_video=true): " . substr($item['video_url'], 0, 100));
                        sendVideo($chatId, $item['video_url'], $storyCaption);
                        $sent = true;
                    } elseif (isset($item['is_video']) && $item['is_video'] === false && isset($item['image_url'])) {
                        error_log("Sending photo (is_video=false): " . substr($item['image_url'], 0, 100));
                        sendPhoto($chatId, $item['image_url'], $storyCaption);
                        $sent = true;
                    }
                    // فرمت قدیمی: video_url مستقیم
                    elseif (isset($item['video_url']) && !empty($item['video_url'])) {
                        error_log("Sending video (video_url): " . substr($item['video_url'], 0, 100));
                        sendVideo($chatId, $item['video_url'], $storyCaption);
                        $sent = true;
                    } elseif (isset($item['image_url']) && !empty($item['image_url'])) {
                        error_log("Sending photo (image_url): " . substr($item['image_url'], 0, 100));
                        sendPhoto($chatId, $item['image_url'], $storyCaption);
                        $sent = true;
                    } elseif (isset($item['display_url']) && !empty($item['display_url'])) {
                        error_log("Sending photo (display_url): " . substr($item['display_url'], 0, 100));
                        sendPhoto($chatId, $item['display_url'], $storyCaption);
                        $sent = true;
                    }
                    
                    // اگر فایل ارسال نشد، پیام خطا بده
                    if (!$sent) {
                        error_log("Story item NOT sent - no valid URL found. Item keys: " . implode(', ', array_keys($item)));
                        sendMessage($chatId, "⚠️ خطا در ارسال استوری " . ($index + 1) . "\n\nلطفاً با پشتیبانی تماس بگیرید.");
                    }
                    
                    usleep(500000); // 500ms تاخیر
                }
                
                // ثبت آمار استفاده
                handleStats($userId, 'download', 'story');
                return;
            }
        }
        
        // استراتژی 2: اگر Fast-Creat موفق نبود و username داریم، از Instagram120 امتحان کنیم
        if (!empty($username)) {
            error_log("Strategy 2: Fast-Creat failed, trying Instagram120 for story: $username");
            $result = getStoriesFromInstagram120($username);
        
            error_log("Story API result for link: " . json_encode($result, JSON_UNESCAPED_UNICODE));
            
            if ($result['success']) {
                $data = $result['data'];
                $items = [];
                
                // فرمت Instagram120: {status: "success", data: {items: [...]}}
                if (isset($data['status']) && $data['status'] === 'success' && isset($data['data']['items'])) {
                    $items = $data['data']['items'];
                }
                // فرمت RapidAPI: {data: {items: [...]}}
                elseif (isset($data['data']) && isset($data['data']['items'])) {
                    $items = $data['data']['items'];
                }
                
                if (!empty($items)) {
                    sendMessage($chatId, "⏳ در حال دانلود و ارسال استوری @{$username}...");
                    
                    foreach ($items as $index => $item) {
                        error_log("Instagram120 Story item " . ($index + 1) . ": " . json_encode($item, JSON_UNESCAPED_UNICODE));
                        
                        // کپشن ساده
                        $storyCaption = "📥 " . BOT_USERNAME;
                        
                        $sent = false;
                        
                        // پشتیبانی از فرمت‌های مختلف
                        if (isset($item['video_url']) && !empty($item['video_url'])) {
                            error_log("Sending video (Instagram120): " . $item['video_url']);
                            sendVideo($chatId, $item['video_url'], $storyCaption);
                            $sent = true;
                        } elseif (isset($item['display_url']) && !empty($item['display_url'])) {
                            error_log("Sending photo (Instagram120): " . $item['display_url']);
                            sendPhoto($chatId, $item['display_url'], $storyCaption);
                            $sent = true;
                        } elseif (isset($item['video_versions']) && !empty($item['video_versions'])) {
                            error_log("Sending video (Instagram120 video_versions): " . $item['video_versions'][0]['url']);
                            sendVideo($chatId, $item['video_versions'][0]['url'], $storyCaption);
                            $sent = true;
                        } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
                            error_log("Sending photo (Instagram120 image_versions2): " . $item['image_versions2']['candidates'][0]['url']);
                            sendPhoto($chatId, $item['image_versions2']['candidates'][0]['url'], $storyCaption);
                            $sent = true;
                        }
                        
                        if (!$sent) {
                            error_log("Instagram120 story NOT sent. Item keys: " . implode(', ', array_keys($item)));
                            sendMessage($chatId, "⚠️ خطا در ارسال استوری " . ($index + 1));
                        }
                        
                        usleep(500000);
                    }
                    
                    // ثبت آمار استفاده
                    handleStats($userId, 'download', 'story');
                    return;
                }
            }
            
            // استراتژی 3: اگر Instagram120 هم موفق نبود، از RapidAPI امتحان کنیم
            if (RAPIDAPI_KEY !== 'YOUR_RAPIDAPI_KEY_HERE') {
                error_log("Strategy 3: Trying RapidAPI for story: $username");
                $result = getStoriesFromRapidAPI($username);
                
                if ($result['success'] && isset($result['data']['data']['items']) && !empty($result['data']['data']['items'])) {
                    $items = $result['data']['data']['items'];
                    sendMessage($chatId, "⏳ در حال دانلود و ارسال استوری @{$username}...");
                    
                    foreach ($items as $index => $item) {
                        // کپشن ساده
                        $storyCaption = "📥 " . BOT_USERNAME;
                        
                        if (isset($item['video_versions']) && !empty($item['video_versions'])) {
                            sendVideo($chatId, $item['video_versions'][0]['url'], $storyCaption);
                        } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
                            sendPhoto($chatId, $item['image_versions2']['candidates'][0]['url'], $storyCaption);
                        }
                        usleep(500000);
                    }
                    
                    // ثبت آمار استفاده
                    handleStats($userId, 'download', 'story');
                    return;
                }
            }
        }
        
        // همه API ها موفق نبودند
        $errorMessage = "❌ <b>خطا در دریافت استوری</b>\n\n";
        $errorMessage .= "دلایل احتمالی:\n";
        $errorMessage .= "• استوری منقضی شده (بیش از 24 ساعت)\n";
        $errorMessage .= "• استوری حذف شده است\n";
        $errorMessage .= "• حساب خصوصی است\n";
        $errorMessage .= "• لینک نامعتبر است";
        sendMessage($chatId, $errorMessage);
        return;
    } else {
        // احتمالاً لینک پروفایل
        $username = extractUsernameFromLink($link);
        
        if ($username) {
            // دریافت اطلاعات پروفایل
            getInstagramProfile($username, $chatId, $userId);
        } else {
            // لینک نامعتبر
            $errorMessage = "❌ <b>لینک نامعتبر</b>\n\n";
            $errorMessage .= "لینک ارسالی قابل پردازش نیست. لطفاً یک لینک معتبر اینستاگرام ارسال کنید.";
            sendMessage($chatId, $errorMessage);
        }
    }
}

/**
 * استخراج نام کاربری از لینک پروفایل
 * 
 * @param string $link لینک پروفایل
 * @return string|false نام کاربری یا false در صورت عدم موفقیت
 */
function extractUsernameFromLink($link) {
    // الگوی استخراج نام کاربری از لینک
    if (preg_match('/instagram\.com\/([a-zA-Z0-9._]{1,30})\/?/', $link, $matches)) {
        return $matches[1];
    }
    
    return false;
}

