<?php
/**
 * مدیریت اکشن‌های پنل ادمین
 */

require_once __DIR__ . '/functions.php';

// فایل ذخیره session های موقت
define('USER_SESSIONS_FILE', __DIR__ . '/user_sessions.json');

/**
 * ذخیره session کاربر
 */
function saveUserSession($userId, $state, $data = []) {
    $sessions = [];
    if (file_exists(USER_SESSIONS_FILE)) {
        $sessions = json_decode(file_get_contents(USER_SESSIONS_FILE), true) ?? [];
    }
    
    $sessions[$userId] = [
        'state' => $state,
        'data' => $data,
        'timestamp' => time()
    ];
    
    file_put_contents(USER_SESSIONS_FILE, json_encode($sessions, JSON_UNESCAPED_UNICODE));
}

/**
 * دریافت session کاربر
 */
function getUserSession($userId) {
    if (!file_exists(USER_SESSIONS_FILE)) {
        return null;
    }
    
    $sessions = json_decode(file_get_contents(USER_SESSIONS_FILE), true) ?? [];
    
    if (!isset($sessions[$userId])) {
        return null;
    }
    
    // پاک کردن session های قدیمی (بیشتر از 1 ساعت)
    if (time() - $sessions[$userId]['timestamp'] > 3600) {
        unset($sessions[$userId]);
        file_put_contents(USER_SESSIONS_FILE, json_encode($sessions, JSON_UNESCAPED_UNICODE));
        return null;
    }
    
    return $sessions[$userId];
}

/**
 * حذف session کاربر
 */
function clearUserSession($userId) {
    if (!file_exists(USER_SESSIONS_FILE)) {
        return;
    }
    
    $sessions = json_decode(file_get_contents(USER_SESSIONS_FILE), true) ?? [];
    
    if (isset($sessions[$userId])) {
        unset($sessions[$userId]);
        file_put_contents(USER_SESSIONS_FILE, json_encode($sessions, JSON_UNESCAPED_UNICODE));
    }
}

/**
 * پردازش اکشن‌های پنل ادمین
 */
function handleAdminAction($callbackQuery) {
    $chatId = $callbackQuery['message']['chat']['id'];
    $userId = $callbackQuery['from']['id'];
    $messageId = $callbackQuery['message']['message_id'];
    $data = $callbackQuery['data'];
    
    // بررسی دسترسی ادمین
    if (!isAdmin($userId)) {
        answerCallbackQuery($callbackQuery['id'], "⛔ شما دسترسی ادمین ندارید!", true);
        return;
    }
    
    $adminAction = str_replace('admin_', '', $data);
    $adminAction = trim($adminAction); // حذف فضاهای خالی اضافی
    
    // چک برای صفحه‌بندی - اگر شامل عدد بود، برای switch ساده‌اش می‌کنیم
    $originalAdminAction = $adminAction;
    if (strpos($adminAction, 'listusers_') === 0) {
        $adminAction = 'listusers';
    }
    
    error_log("handleAdminAction - data: '$data', adminAction after processing: '$adminAction'");
    
    switch ($adminAction) {
        case 'stats':
            // آمار کاربران
            $stats = getUserStats();
            $statsMessage = "<b>📊 آمار ربات</b>\n\n";
            $statsMessage .= "<b>👥 کاربران:</b>\n";
            $statsMessage .= "• تعداد کل: " . number_format($stats['total_users']) . "\n";
            $statsMessage .= "• کاربران فعال امروز: " . number_format($stats['active_today']) . "\n";
            $statsMessage .= "• کاربران جدید امروز: " . number_format($stats['new_users_today']) . "\n\n";
            
            $statsMessage .= "<b>📥 دانلودها:</b>\n";
            $statsMessage .= "• دانلودهای امروز: " . number_format($stats['downloads_today']) . "\n";
            $statsMessage .= "• کل دانلودها: " . number_format($stats['total_downloads']) . "\n\n";
            
            $statsMessage .= "📅 تاریخ: " . date('Y/m/d H:i');
            
            // دکمه بازگشت
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔄 به‌روزرسانی', 'callback_data' => 'admin_stats']],
                    [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']]
                ]
            ];
            
            editMessageText($chatId, $messageId, $statsMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id'], "آمار به‌روزرسانی شد ✅");
            break;
            
        case 'broadcast':
            // ارسال پیام همگانی
            $broadcastMessage = "<b>📢 ارسال پیام همگانی</b>\n\n";
            $broadcastMessage .= "لطفاً پیامی که می‌خواهید برای تمام کاربران ارسال شود را بفرستید.\n\n";
            $broadcastMessage .= "⚠️ توجه: پیام برای تمام کاربران ربات ارسال خواهد شد.\n\n";
            $broadcastMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $broadcastMessage, $keyboard);
            saveUserSession($userId, 'waiting_broadcast_message');
            answerCallbackQuery($callbackQuery['id'], "منتظر پیام شما هستم...");
            break;
            
        case 'forward':
            // فوروارد همگانی
            $forwardMessage = "<b>↪️ فوروارد همگانی</b>\n\n";
            $forwardMessage .= "لطفاً پیامی که می‌خواهید برای تمام کاربران فوروارد شود را ارسال کنید.\n\n";
            $forwardMessage .= "⚠️ توجه: پیام برای تمام کاربران ربات فوروارد خواهد شد.\n\n";
            $forwardMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $forwardMessage, $keyboard);
            saveUserSession($userId, 'waiting_forward_message');
            answerCallbackQuery($callbackQuery['id'], "منتظر پیام شما هستم...");
            break;
            
        case 'addadmin':
            // افزودن ادمین
            $addAdminMessage = "<b>➕ افزودن ادمین جدید</b>\n\n";
            $addAdminMessage .= "لطفاً پیامی از کاربری که می‌خواهید ادمین شود را فوروارد کنید.\n\n";
            $addAdminMessage .= "یا شناسه عددی تلگرام او را ارسال کنید.\n\n";
            $addAdminMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $addAdminMessage, $keyboard);
            saveUserSession($userId, 'waiting_add_admin');
            answerCallbackQuery($callbackQuery['id'], "منتظر اطلاعات کاربر هستم...");
            break;
            
        case 'deladmin':
            // حذف ادمین - نمایش لیست و درخواست ID
            $admins = getAdminsList();
            
            if (empty($admins)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ ادمینی در لیست نیست!", true);
                break;
            }
            
            $adminListMessage = "<b>➖ حذف ادمین</b>\n\n";
            $adminListMessage .= "لیست ادمین‌ها:\n\n";
            
            foreach ($admins as $index => $admin) {
                $adminListMessage .= ($index + 1) . ". ";
                $adminListMessage .= "<b>" . $admin['first_name'];
                
                if (!empty($admin['last_name'])) {
                    $adminListMessage .= " " . $admin['last_name'];
                }
                
                if (!empty($admin['username'])) {
                    $adminListMessage .= "</b> (@" . $admin['username'] . ")";
                } else {
                    $adminListMessage .= "</b>";
                }
                
                $adminListMessage .= "\n   🆔 <code>" . $admin['telegram_id'] . "</code>\n\n";
            }
            
            $adminListMessage .= "━━━━━━━━━━━━━━━\n\n";
            $adminListMessage .= "برای حذف، <b>شناسه عددی (ID)</b> ادمین را ارسال کنید.\n\n";
            $adminListMessage .= "⚠️ ادمین‌های اصلی قابل حذف نیستند.\n\n";
            $adminListMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $adminListMessage, $keyboard);
            saveUserSession($userId, 'waiting_delete_admin');
            answerCallbackQuery($callbackQuery['id'], "منتظر شناسه ادمین هستم...");
            break;
            
        case 'listadmins':
            // لیست ادمین‌ها
            $admins = getAdminsList();
            
            if (empty($admins)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ ادمینی در لیست نیست!", true);
                break;
            }
            
            $adminListMessage = "<b>📋 لیست ادمین‌ها</b>\n\n";
            
            foreach ($admins as $index => $admin) {
                $adminListMessage .= ($index + 1) . ". ";
                $adminListMessage .= $admin['first_name'];
                
                if (!empty($admin['last_name'])) {
                    $adminListMessage .= " " . $admin['last_name'];
                }
                
                if (!empty($admin['username'])) {
                    $adminListMessage .= " (@" . $admin['username'] . ")";
                }
                
                $adminListMessage .= "\n";
                $adminListMessage .= "   🆔 " . $admin['telegram_id'] . "\n";
                $adminListMessage .= "   📅 " . date('Y/m/d', strtotime($admin['added_at'])) . "\n\n";
            }
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']]
                ]
            ];
            
            editMessageText($chatId, $messageId, $adminListMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'addchannel':
            // افزودن کانال - مرحله 1: نام نمایشی
            $addChannelMessage = "<b>➕ افزودن کانال جوین اجباری</b>\n\n";
            $addChannelMessage .= "مرحله 1️⃣ از 3️⃣\n\n";
            $addChannelMessage .= "لطفاً <b>نام نمایشی</b> کانال را وارد کنید:\n\n";
            $addChannelMessage .= "مثال: کانال اصلی ما\n\n";
            $addChannelMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $addChannelMessage, $keyboard);
            saveUserSession($userId, 'waiting_channel_name');
            answerCallbackQuery($callbackQuery['id'], "منتظر نام کانال هستم...");
            break;
            
        case 'delchannel':
            // حذف کانال - نمایش لیست
            $channels = getRequiredChannelsList();
            
            if (empty($channels)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ کانالی در لیست نیست!", true);
                break;
            }
            
            $channelListMessage = "<b>➖ حذف کانال جوین اجباری</b>\n\n";
            $channelListMessage .= "روی کانالی که می‌خواهید حذف کنید کلیک کنید:\n\n";
            
            $keyboard = ['inline_keyboard' => []];
            foreach ($channels as $channel) {
                $keyboard['inline_keyboard'][] = [
                    ['text' => "🗑 " . $channel['channel_name'], 'callback_data' => 'delchannel_' . $channel['channel_username']]
                ];
            }
            
            $keyboard['inline_keyboard'][] = [
                ['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']
            ];
            
            editMessageText($chatId, $messageId, $channelListMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'listchannels':
            // لیست کانال‌ها
            $channels = getRequiredChannelsList();
            
            if (empty($channels)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ کانالی در لیست نیست!", true);
                break;
            }
            
            $channelListMessage = "<b>📋 لیست کانال‌های جوین اجباری</b>\n\n";
            
            foreach ($channels as $index => $channel) {
                $channelListMessage .= ($index + 1) . ". <b>" . $channel['channel_name'] . "</b>\n";
                $channelListMessage .= "   🔗 @" . $channel['channel_username'] . "\n";
                
                if (!empty($channel['channel_id'])) {
                    $channelListMessage .= "   🆔 " . $channel['channel_id'] . "\n";
                }
                
                $channelListMessage .= "   📅 " . date('Y/m/d', strtotime($channel['added_at'])) . "\n\n";
            }
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']]
                ]
            ];
            
            editMessageText($chatId, $messageId, $channelListMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'listusers':
            // لیست کاربران با صفحه‌بندی
            // دریافت شماره صفحه از originalAdminAction (پیش‌فرض صفحه 1)
            $page = 1;
            if (strpos($originalAdminAction, 'listusers_') === 0) {
                $page = intval(str_replace('listusers_', '', $originalAdminAction));
            }
            
            $users = getUsersList();
            
            if (empty($users)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ کاربری در لیست نیست!", true);
                break;
            }
            
            // تنظیمات صفحه‌بندی
            $perPage = 10; // تعداد کاربر در هر صفحه
            $totalUsers = count($users);
            $totalPages = ceil($totalUsers / $perPage);
            
            // اطمینان از معتبر بودن شماره صفحه
            if ($page < 1) $page = 1;
            if ($page > $totalPages) $page = $totalPages;
            
            // محاسبه offset
            $offset = ($page - 1) * $perPage;
            $usersPage = array_slice($users, $offset, $perPage);
            
            $userListMessage = "<b>👥 لیست کاربران</b>\n\n";
            $userListMessage .= "تعداد کل: " . $totalUsers . " کاربر\n\n";
            $userListMessage .= "━━━━━━━━━━━━━━━\n\n";
            
            foreach ($usersPage as $index => $user) {
                $globalIndex = $offset + $index + 1;
                $userListMessage .= $globalIndex . ". ";
                
                // نام
                if (!empty($user['first_name'])) {
                    $userListMessage .= "<b>" . $user['first_name'];
                    if (!empty($user['last_name'])) {
                        $userListMessage .= " " . $user['last_name'];
                    }
                    $userListMessage .= "</b>";
                } else {
                    $userListMessage .= "<b>کاربر</b>";
                }
                
                $userListMessage .= "\n";
                
                // شناسه عددی
                $userListMessage .= "   🆔 ID: " . $user['id'] . "\n";
                
                // شناسه تلگرام
                $userListMessage .= "   📱 تلگرام: <code>" . $user['telegram_id'] . "</code>";
                
                // نام کاربری تلگرام (اگر داره)
                if (!empty($user['username'])) {
                    $userListMessage .= " (@" . $user['username'] . ")";
                }
                $userListMessage .= "\n";
                
                // تعداد دانلود
                $userListMessage .= "   📥 دانلود: " . $user['download_count'] . "\n";
                
                // تاریخ عضویت
                $userListMessage .= "   📅 عضویت: " . date('Y/m/d', strtotime($user['joined_at'])) . "\n\n";
            }
            
            // ایجاد دکمه‌های صفحه‌بندی
            $keyboard = ['inline_keyboard' => []];
            
            // ردیف اول: دکمه‌های قبلی و بعدی + شماره صفحه
            $navRow = [];
            
            if ($page > 1) {
                $navRow[] = ['text' => '◀️ قبلی', 'callback_data' => 'admin_listusers_' . ($page - 1)];
            }
            
            $navRow[] = ['text' => "📄 $page/$totalPages", 'callback_data' => 'admin_noop'];
            
            if ($page < $totalPages) {
                $navRow[] = ['text' => 'بعدی ▶️', 'callback_data' => 'admin_listusers_' . ($page + 1)];
            }
            
            $keyboard['inline_keyboard'][] = $navRow;
            
            // ردیف دوم: به‌روزرسانی و بازگشت
            $keyboard['inline_keyboard'][] = [
                ['text' => '🔄 به‌روزرسانی', 'callback_data' => 'admin_listusers_' . $page],
                ['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']
            ];
            
            editMessageText($chatId, $messageId, $userListMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id'], "صفحه $page از $totalPages");
            break;
            
        case 'setstart':
            // تنظیم متن استارت
            $settings = getBotSettingsFromFile();
            $currentMessage = $settings['start_message'] ?? '';
            
            $setStartMessage = "<b>📝 تنظیم متن استارت</b>\n\n";
            $setStartMessage .= "<b>متن فعلی:</b>\n";
            $setStartMessage .= "<code>" . htmlspecialchars($currentMessage) . "</code>\n\n";
            $setStartMessage .= "━━━━━━━━━━━━━━━\n\n";
            $setStartMessage .= "لطفاً متن جدید استارت را ارسال کنید.\n\n";
            $setStartMessage .= "<b>متغیرهای قابل استفاده:</b>\n";
            $setStartMessage .= "• <code>{first_name}</code> - نام کاربر\n";
            $setStartMessage .= "• <code>{total_users}</code> - تعداد کل کاربران\n";
            $setStartMessage .= "• <code>{total_downloads}</code> - تعداد کل دانلودها\n\n";
            $setStartMessage .= "⚠️ از تگ‌های HTML استفاده کنید: &lt;b&gt;، &lt;i&gt;، &lt;code&gt;\n\n";
            $setStartMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $setStartMessage, $keyboard);
            saveUserSession($userId, 'waiting_start_message');
            answerCallbackQuery($callbackQuery['id'], "منتظر متن استارت جدید هستم...");
            break;
            
        case 'buttons':
            // مدیریت دکمه‌ها
            $settings = getBotSettingsFromFile();
            $buttons = $settings['buttons'] ?? [];
            
            $buttonsMessage = "<b>🔘 مدیریت دکمه‌های ربات</b>\n\n";
            $buttonsMessage .= "<b>دکمه‌های فعلی:</b>\n\n";
            
            $buttonIndex = 1;
            foreach ($buttons as $row) {
                foreach ($row as $button) {
                    $buttonsMessage .= $buttonIndex . ". " . $button['text'] . "\n";
                    $buttonIndex++;
                }
            }
            
            $buttonsMessage .= "\n━━━━━━━━━━━━━━━\n\n";
            $buttonsMessage .= "برای مشاهده و ویرایش، از دکمه‌های زیر استفاده کنید:";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '➕ افزودن دکمه', 'callback_data' => 'admin_addbutton']
                    ],
                    [
                        ['text' => '➖ حذف دکمه', 'callback_data' => 'admin_delbutton']
                    ],
                    [
                        ['text' => '♻️ بازنشانی پیش‌فرض', 'callback_data' => 'admin_resetbuttons']
                    ],
                    [
                        ['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']
                    ]
                ]
            ];
            
            editMessageText($chatId, $messageId, $buttonsMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'addbutton':
            // افزودن دکمه جدید - انتخاب نوع
            $addButtonMessage = "<b>➕ افزودن دکمه جدید</b>\n\n";
            $addButtonMessage .= "چه نوع دکمه‌ای می‌خواهید اضافه کنید؟\n\n";
            $addButtonMessage .= "🔹 <b>دکمه با متن پاسخ:</b> وقتی روی دکمه کلیک شود، یک متن سفارشی نمایش داده می‌شود\n\n";
            $addButtonMessage .= "🔹 <b>دکمه با لینک:</b> وقتی روی دکمه کلیک شود، یک لینک باز می‌شود (مثل سفارش ربات)\n\n";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '💬 دکمه با متن پاسخ', 'callback_data' => 'admin_addbutton_text']
                    ],
                    [
                        ['text' => '🔗 دکمه با لینک', 'callback_data' => 'admin_addbutton_url']
                    ],
                    [
                        ['text' => '❌ لغو', 'callback_data' => 'admin_cancel']
                    ]
                ]
            ];
            
            editMessageText($chatId, $messageId, $addButtonMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'addbutton_text':
            // افزودن دکمه با متن پاسخ - مرحله 1: متن دکمه
            $addButtonMessage = "<b>➕ افزودن دکمه با متن پاسخ</b>\n\n";
            $addButtonMessage .= "مرحله 1️⃣ از 2️⃣\n\n";
            $addButtonMessage .= "لطفاً <b>متن دکمه</b> را ارسال کنید:\n\n";
            $addButtonMessage .= "مثال: 📱 تماس با پشتیبانی\n\n";
            $addButtonMessage .= "برای لغو: /cancel یا /panel";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $addButtonMessage, $keyboard);
            saveUserSession($userId, 'waiting_button_text_type', ['type' => 'text']);
            answerCallbackQuery($callbackQuery['id'], "منتظر متن دکمه هستم...");
            break;
            
        case 'addbutton_url':
            // افزودن دکمه با URL - مرحله 1: متن دکمه
            $addButtonMessage = "<b>➕ افزودن دکمه با لینک</b>\n\n";
            $addButtonMessage .= "مرحله 1️⃣ از 2️⃣\n\n";
            $addButtonMessage .= "لطفاً <b>متن دکمه</b> را ارسال کنید:\n\n";
            $addButtonMessage .= "مثال: سفارش ربات | خرید VPN\n\n";
            $addButtonMessage .= "برای لغو: /cancel یا /panel";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $addButtonMessage, $keyboard);
            saveUserSession($userId, 'waiting_button_text_type', ['type' => 'url']);
            answerCallbackQuery($callbackQuery['id'], "منتظر متن دکمه هستم...");
            break;
            
        case 'delbutton':
            // حذف دکمه
            $settings = getBotSettingsFromFile();
            $buttons = $settings['buttons'] ?? [];
            
            if (empty($buttons)) {
                answerCallbackQuery($callbackQuery['id'], "هیچ دکمه‌ای وجود ندارد!", true);
                break;
            }
            
            $delButtonMessage = "<b>➖ حذف دکمه</b>\n\n";
            $delButtonMessage .= "روی دکمه‌ای که می‌خواهید حذف کنید کلیک کنید:\n\n";
            
            $keyboard = ['inline_keyboard' => []];
            $buttonIndex = 0;
            foreach ($buttons as $rowIndex => $row) {
                foreach ($row as $colIndex => $button) {
                    $keyboard['inline_keyboard'][] = [
                        ['text' => "🗑 " . $button['text'], 'callback_data' => 'delbutton_' . $rowIndex . '_' . $colIndex]
                    ];
                    $buttonIndex++;
                }
            }
            
            $keyboard['inline_keyboard'][] = [
                ['text' => '🔙 بازگشت', 'callback_data' => 'admin_buttons']
            ];
            
            editMessageText($chatId, $messageId, $delButtonMessage, $keyboard);
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'resetbuttons':
            // بازنشانی دکمه‌ها به حالت پیش‌فرض
            $defaultSettings = [
                'buttons' => [
                    [
                        ['text' => '📥 دانلود پست/ریل'],
                        ['text' => '📹 دانلود استوری']
                    ],
                    [
                        ['text' => '❓ راهنما']
                    ],
                    [
                        ['text' => 'سفارش ربات | خرید VPN']
                    ]
                ]
            ];
            
            $settings = getBotSettingsFromFile();
            $settings['buttons'] = $defaultSettings['buttons'];
            saveBotSettingsToFile($settings);
            
            answerCallbackQuery($callbackQuery['id'], "✅ دکمه‌ها به حالت پیش‌فرض بازگشتند", true);
            
            // بازگشت به منوی مدیریت دکمه‌ها
            handleAdminAction(array_merge($callbackQuery, ['data' => 'admin_buttons']));
            break;
            
        case 'noop':
            // عملیات خالی - برای دکمه‌هایی که فقط نمایشی هستند
            answerCallbackQuery($callbackQuery['id']);
            break;
            
        case 'back':
        case 'cancel':
            // بازگشت به منوی اصلی یا لغو
            clearUserSession($userId);
            require_once __DIR__ . '/admin.php';
            handleAdminPanel($chatId);
            answerCallbackQuery($callbackQuery['id'], "✅ عملیات لغو شد");
            break;
            
        default:
            // لاگ برای دیباگ
            error_log("Admin action default case - adminAction: " . $adminAction . ", original data: " . $data);
            
            // بررسی اکشن‌های حذف
            if (strpos($adminAction, 'delchannel_') === 0) {
                $channelUsername = str_replace('delchannel_', '', $adminAction);
                error_log("Attempting to delete channel: " . $channelUsername);
                
                if (removeRequiredChannel($channelUsername)) {
                    error_log("Channel deleted successfully: " . $channelUsername);
                    answerCallbackQuery($callbackQuery['id'], "✅ کانال با موفقیت حذف شد", true);
                    
                    // بازگشت به لیست
                    handleAdminAction(array_merge($callbackQuery, ['data' => 'admin_delchannel']));
                } else {
                    error_log("Failed to delete channel: " . $channelUsername);
                    answerCallbackQuery($callbackQuery['id'], "❌ خطا در حذف کانال", true);
                }
            } elseif ($adminAction === 'addbutton_text' || $adminAction === 'addbutton_url') {
                // این case ها در بالا handle شدن، اینجا نباید برسن
                answerCallbackQuery($callbackQuery['id'], "در حال پردازش...");
            } elseif (strpos($adminAction, 'delbutton_') === 0) {
                // حذف دکمه
                $parts = explode('_', $adminAction);
                if (count($parts) === 3) {
                    $rowIndex = intval($parts[1]);
                    $colIndex = intval($parts[2]);
                    
                    $settings = getBotSettingsFromFile();
                    $buttons = $settings['buttons'] ?? [];
                    
                    if (isset($buttons[$rowIndex][$colIndex])) {
                        $buttonText = $buttons[$rowIndex][$colIndex]['text'];
                        
                        // حذف دکمه
                        unset($buttons[$rowIndex][$colIndex]);
                        
                        // اگر row خالی شد، حذفش کن
                        $buttons[$rowIndex] = array_values($buttons[$rowIndex]);
                        if (empty($buttons[$rowIndex])) {
                            unset($buttons[$rowIndex]);
                        }
                        
                        // ریندکس کردن آرایه
                        $buttons = array_values($buttons);
                        
                        // ذخیره
                        $settings['buttons'] = $buttons;
                        $saveResult = saveBotSettingsToFile($settings);
                        
                        error_log("Button deleted: $buttonText, Save result: " . ($saveResult ? 'success' : 'failed'));
                        error_log("Buttons after delete: " . json_encode($buttons, JSON_UNESCAPED_UNICODE));
                        
                        // نمایش لیست به‌روزرسانی شده
                        if (empty($buttons)) {
                            $delButtonMessage = "<b>➖ حذف دکمه</b>\n\n";
                            $delButtonMessage .= "✅ دکمه حذف شد: <b>" . $buttonText . "</b>\n\n";
                            $delButtonMessage .= "⚠️ هیچ دکمه‌ای باقی نمانده است!\n\n";
                            $delButtonMessage .= "می‌توانید دکمه‌های جدید اضافه کنید یا به حالت پیش‌فرض بازگردانید.";
                            
                            $keyboard = [
                                'inline_keyboard' => [
                                    [['text' => '🔙 بازگشت', 'callback_data' => 'admin_buttons']]
                                ]
                            ];
                            
                            editMessageText($chatId, $messageId, $delButtonMessage, $keyboard);
                            answerCallbackQuery($callbackQuery['id'], "✅ دکمه حذف شد: " . $buttonText);
                        } else {
                            // نمایش لیست جدید دکمه‌ها برای حذف
                            $delButtonMessage = "<b>➖ حذف دکمه</b>\n\n";
                            $delButtonMessage .= "✅ دکمه حذف شد: <b>" . $buttonText . "</b>\n\n";
                            $delButtonMessage .= "روی دکمه دیگری که می‌خواهید حذف کنید کلیک کنید:\n\n";
                            
                            $keyboard = ['inline_keyboard' => []];
                            foreach ($buttons as $rowIndex => $row) {
                                foreach ($row as $colIndex => $button) {
                                    $keyboard['inline_keyboard'][] = [
                                        ['text' => "🗑 " . $button['text'], 'callback_data' => 'delbutton_' . $rowIndex . '_' . $colIndex]
                                    ];
                                }
                            }
                            
                            $keyboard['inline_keyboard'][] = [
                                ['text' => '🔙 بازگشت', 'callback_data' => 'admin_buttons']
                            ];
                            
                            editMessageText($chatId, $messageId, $delButtonMessage, $keyboard);
                            answerCallbackQuery($callbackQuery['id'], "✅ دکمه حذف شد: " . $buttonText);
                        }
                    } else {
                        answerCallbackQuery($callbackQuery['id'], "❌ دکمه یافت نشد!", true);
                    }
                } else {
                    answerCallbackQuery($callbackQuery['id'], "❌ خطا در پردازش!", true);
                }
            } else {
                error_log("Unknown admin action: " . $adminAction);
                answerCallbackQuery($callbackQuery['id'], "این قابلیت هنوز پیاده‌سازی نشده است.");
            }
            break;
    }
}

/**
 * پردازش پیام‌های کاربران در حالت انتظار
 */
function handleUserSessionMessage($message) {
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $text = $message['text'] ?? '';
    
    $session = getUserSession($userId);
    
    if (!$session) {
        return false; // هیچ session فعالی نیست
    }
    
    // بررسی دسترسی ادمین
    if (!isAdmin($userId)) {
        clearUserSession($userId);
        return false;
    }
    
    $state = $session['state'];
    $data = $session['data'];
    
    switch ($state) {
        case 'waiting_broadcast_message':
            // ارسال پیام همگانی
            sendMessage($chatId, "⏳ در حال ارسال پیام به کاربران...\n\nلطفاً صبر کنید...");
            
            $result = sendBroadcast($text, null, null, $userId, false);
            
            $resultMessage = "✅ <b>ارسال پیام همگانی انجام شد</b>\n\n";
            $resultMessage .= "📊 تعداد کل کاربران: " . $result['total'] . "\n";
            $resultMessage .= "✅ ارسال موفق: " . $result['successful'] . "\n";
            $resultMessage .= "❌ ارسال ناموفق: " . $result['failed'];
            
            sendMessage($chatId, $resultMessage);
            clearUserSession($userId);
            return true;
            
        case 'waiting_forward_message':
            // فوروارد همگانی
            $messageId = $message['message_id'] ?? null;
            
            if (!$messageId) {
                sendMessage($chatId, "❌ خطا در دریافت پیام. لطفاً دوباره تلاش کنید.");
                clearUserSession($userId);
                return true;
            }
            
            sendMessage($chatId, "⏳ در حال فوروارد پیام به کاربران...\n\nلطفاً صبر کنید...");
            
            $result = sendBroadcast('', $chatId, $messageId, $userId, true);
            
            $resultMessage = "✅ <b>فوروارد همگانی انجام شد</b>\n\n";
            $resultMessage .= "📊 تعداد کل کاربران: " . $result['total'] . "\n";
            $resultMessage .= "✅ فوروارد موفق: " . $result['successful'] . "\n";
            $resultMessage .= "❌ فوروارد ناموفق: " . $result['failed'];
            
            sendMessage($chatId, $resultMessage);
            clearUserSession($userId);
            return true;
            
        case 'waiting_add_admin':
            // افزودن ادمین
            $targetId = null;
            $targetUsername = '';
            $targetFirstName = '';
            $targetLastName = '';
            
            // بررسی اینکه آیا پیام فوروارد شده است
            if (isset($message['forward_from'])) {
                $forwardedUser = $message['forward_from'];
                $targetId = $forwardedUser['id'];
                $targetUsername = $forwardedUser['username'] ?? '';
                $targetFirstName = $forwardedUser['first_name'] ?? '';
                $targetLastName = $forwardedUser['last_name'] ?? '';
            } elseif (is_numeric($text)) {
                $targetId = intval($text);
                
                // سعی کنیم اطلاعات را از دیتابیس بگیریم
                $conn = getDbConnection();
                $stmt = $conn->prepare("SELECT username, first_name, last_name FROM users WHERE telegram_id = ?");
                $stmt->bind_param("i", $targetId);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result->num_rows > 0) {
                    $userData = $result->fetch_assoc();
                    $targetUsername = $userData['username'] ?? '';
                    $targetFirstName = $userData['first_name'] ?? 'کاربر';
                    $targetLastName = $userData['last_name'] ?? '';
                }
                
                $stmt->close();
                $conn->close();
            } else {
                sendMessage($chatId, "❌ لطفاً یک پیام از کاربر مورد نظر فوروارد کنید یا شناسه عددی او را ارسال کنید.");
                clearUserSession($userId);
                return true;
            }
            
            if (!$targetId) {
                sendMessage($chatId, "❌ شناسه کاربر نامعتبر است.");
                clearUserSession($userId);
                return true;
            }
            
            // بررسی اینکه آیا این کاربر خود ادمین اصلی است
            if ($targetId == $userId) {
                sendMessage($chatId, "⚠️ شما خودتان از قبل ادمین هستید!");
                clearUserSession($userId);
                return true;
            }
            
            $addResult = addAdmin($targetId, $targetUsername, $targetFirstName, $targetLastName, $userId);
            
            if ($addResult) {
                $successMessage = "✅ <b>ادمین جدید با موفقیت اضافه شد</b>\n\n";
                $successMessage .= "🆔 شناسه: <code>" . $targetId . "</code>\n";
                
                if (!empty($targetFirstName)) {
                    $successMessage .= "👤 نام: " . $targetFirstName;
                    if (!empty($targetLastName)) {
                        $successMessage .= " " . $targetLastName;
                    }
                    $successMessage .= "\n";
                }
                
                if (!empty($targetUsername)) {
                    $successMessage .= "📱 نام کاربری: @" . $targetUsername . "\n";
                }
                
                $successMessage .= "\n✨ این کاربر اکنون می‌تواند از پنل مدیریت استفاده کند.";
                
                sendMessage($chatId, $successMessage);
                
                // ثبت لاگ
                error_log("Admin added: $targetId by $userId");
            } else {
                $errorMessage = "❌ <b>خطا در افزودن ادمین</b>\n\n";
                $errorMessage .= "دلایل احتمالی:\n";
                $errorMessage .= "• این کاربر از قبل ادمین است\n";
                $errorMessage .= "• خطا در ارتباط با دیتابیس\n\n";
                $errorMessage .= "🆔 شناسه ارسالی: <code>" . $targetId . "</code>";
                
                sendMessage($chatId, $errorMessage);
                
                // ثبت لاگ
                error_log("Failed to add admin: $targetId by $userId");
            }
            
            clearUserSession($userId);
            return true;
            
        case 'waiting_delete_admin':
            // حذف ادمین بر اساس ID
            if (!is_numeric($text)) {
                sendMessage($chatId, "❌ لطفاً فقط شناسه عددی (ID) ادمین را ارسال کنید.");
                return true;
            }
            
            $targetId = intval($text);
            
            // بررسی اینکه ادمین اصلی نباشه
            $mainAdminIds = explode(',', ADMIN_IDS);
            $isMainAdmin = false;
            foreach ($mainAdminIds as $mainId) {
                if (trim($mainId) == $targetId) {
                    $isMainAdmin = true;
                    break;
                }
            }
            
            if ($isMainAdmin) {
                sendMessage($chatId, "⛔ <b>ادمین اصلی قابل حذف نیست!</b>\n\n🆔 شناسه: <code>" . $targetId . "</code>");
                clearUserSession($userId);
                return true;
            }
            
            // حذف ادمین
            if (removeAdmin($targetId)) {
                $successMessage = "✅ <b>ادمین با موفقیت حذف شد</b>\n\n";
                $successMessage .= "🆔 شناسه: <code>" . $targetId . "</code>\n\n";
                $successMessage .= "این کاربر دیگر دسترسی به پنل مدیریت ندارد.";
                
                sendMessage($chatId, $successMessage);
                error_log("Admin deleted: $targetId by $userId");
            } else {
                $errorMessage = "❌ <b>خطا در حذف ادمین</b>\n\n";
                $errorMessage .= "دلایل احتمالی:\n";
                $errorMessage .= "• این شناسه در لیست ادمین‌ها نیست\n";
                $errorMessage .= "• خطا در ارتباط با دیتابیس\n\n";
                $errorMessage .= "🆔 شناسه ارسالی: <code>" . $targetId . "</code>";
                
                sendMessage($chatId, $errorMessage);
                error_log("Failed to delete admin: $targetId by $userId");
            }
            
            clearUserSession($userId);
            return true;
            
        case 'waiting_channel_name':
            // مرحله 2: دریافت لینک دعوت
            $channelName = $text;
            
            $inviteLinkMessage = "<b>➕ افزودن کانال جوین اجباری</b>\n\n";
            $inviteLinkMessage .= "مرحله 2️⃣ از 3️⃣\n\n";
            $inviteLinkMessage .= "نام کانال: <b>" . $channelName . "</b>\n\n";
            $inviteLinkMessage .= "لطفاً <b>لینک دعوت</b> کانال را ارسال کنید:\n\n";
            $inviteLinkMessage .= "برای کانال‌های عمومی: https://t.me/channel_name\n";
            $inviteLinkMessage .= "برای کانال‌های خصوصی: https://t.me/+AbCdEfG...\n\n";
            $inviteLinkMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $inviteLinkMessage, $keyboard);
            saveUserSession($userId, 'waiting_channel_link', ['channel_name' => $channelName]);
            return true;
            
        case 'waiting_channel_link':
            // مرحله 3: فوروارد پست از کانال
            $inviteLink = $text;
            $channelName = $data['channel_name'];
            
            $forwardPostMessage = "<b>➕ افزودن کانال جوین اجباری</b>\n\n";
            $forwardPostMessage .= "مرحله 3️⃣ از 3️⃣\n\n";
            $forwardPostMessage .= "نام کانال: <b>" . $channelName . "</b>\n";
            $forwardPostMessage .= "لینک دعوت: <code>" . $inviteLink . "</code>\n\n";
            $forwardPostMessage .= "لطفاً یک <b>پست از کانال</b> را فوروارد کنید تا شناسه کانال ثبت شود.\n\n";
            $forwardPostMessage .= "برای لغو: /cancel یا /panel";
            
            // افزودن دکمه لغو
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                ]
            ];
            
            sendMessage($chatId, $forwardPostMessage, $keyboard);
            saveUserSession($userId, 'waiting_channel_forward', [
                'channel_name' => $channelName,
                'invite_link' => $inviteLink
            ]);
            return true;
            
        case 'waiting_channel_forward':
            // تکمیل افزودن کانال
            if (!isset($message['forward_from_chat'])) {
                sendMessage($chatId, "❌ لطفاً یک پست از کانال را فوروارد کنید.");
                return true;
            }
            
            $forwardedChat = $message['forward_from_chat'];
            $channelId = $forwardedChat['id'];
            $channelUsername = $forwardedChat['username'] ?? '';
            $channelName = $data['channel_name'];
            $inviteLink = $data['invite_link'];
            
            // استخراج username از لینک اگر در فوروارد نبود
            if (empty($channelUsername)) {
                if (preg_match('/t\.me\/([a-zA-Z0-9_]+)/', $inviteLink, $matches)) {
                    $channelUsername = $matches[1];
                } else {
                    // برای کانال‌های خصوصی از ID استفاده می‌کنیم
                    $channelUsername = 'private_' . abs($channelId);
                }
            }
            
            if (addRequiredChannel($channelUsername, $channelName, $channelId, $userId, $inviteLink)) {
                $successMessage = "✅ <b>کانال با موفقیت اضافه شد</b>\n\n";
                $successMessage .= "📢 نام: <b>" . $channelName . "</b>\n";
                $successMessage .= "🔗 نام کاربری: @" . $channelUsername . "\n";
                $successMessage .= "🆔 شناسه: " . $channelId . "\n";
                $successMessage .= "🔗 لینک دعوت: " . $inviteLink;
                
                sendMessage($chatId, $successMessage);
            } else {
                sendMessage($chatId, "❌ این کانال از قبل در لیست است یا خطایی رخ داده است.");
            }
            
            clearUserSession($userId);
            return true;
            
        case 'waiting_start_message':
            // تنظیم متن استارت
            $newStartMessage = $text;
            
            $settings = getBotSettingsFromFile();
            $settings['start_message'] = $newStartMessage;
            
            if (saveBotSettingsToFile($settings)) {
                $successMessage = "✅ <b>متن استارت با موفقیت تنظیم شد</b>\n\n";
                $successMessage .= "<b>متن جدید:</b>\n";
                $successMessage .= "<code>" . htmlspecialchars($newStartMessage) . "</code>\n\n";
                $successMessage .= "برای مشاهده نتیجه، دستور /start را ارسال کنید.";
                
                sendMessage($chatId, $successMessage);
            } else {
                sendMessage($chatId, "❌ خطا در ذخیره تنظیمات. لطفاً دوباره تلاش کنید.");
            }
            
            clearUserSession($userId);
            return true;
            
        case 'waiting_button_text_type':
            // دریافت متن دکمه - مرحله 1
            $buttonText = $text;
            $buttonType = $data['type'];
            
            if ($buttonType === 'text') {
                // دکمه با متن پاسخ - مرحله 2: دریافت متن پاسخ
                $responseMessage = "<b>➕ افزودن دکمه با متن پاسخ</b>\n\n";
                $responseMessage .= "مرحله 2️⃣ از 2️⃣\n\n";
                $responseMessage .= "متن دکمه: <b>" . $buttonText . "</b>\n\n";
                $responseMessage .= "حالا <b>متن پاسخی</b> که می‌خواهید وقتی روی دکمه کلیک شود نمایش داده شود را ارسال کنید:\n\n";
                $responseMessage .= "مثال: برای تماس با پشتیبانی به @support پیام دهید\n\n";
                $responseMessage .= "⚠️ می‌توانید از تگ‌های HTML استفاده کنید: &lt;b&gt;، &lt;i&gt;، &lt;code&gt;\n\n";
                $responseMessage .= "برای لغو: /cancel یا /panel";
                
                $keyboard = [
                    'inline_keyboard' => [
                        [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                    ]
                ];
                
                sendMessage($chatId, $responseMessage, $keyboard);
                saveUserSession($userId, 'waiting_button_response_text', [
                    'button_text' => $buttonText,
                    'type' => 'text'
                ]);
            } elseif ($buttonType === 'url') {
                // دکمه با URL - مرحله 2: دریافت URL
                $urlMessage = "<b>➕ افزودن دکمه با لینک</b>\n\n";
                $urlMessage .= "مرحله 2️⃣ از 2️⃣\n\n";
                $urlMessage .= "متن دکمه: <b>" . $buttonText . "</b>\n\n";
                $urlMessage .= "حالا <b>لینک (URL)</b> را ارسال کنید:\n\n";
                $urlMessage .= "مثال: https://t.me/amirth1998\n\n";
                $urlMessage .= "⚠️ لینک باید با http:// یا https:// شروع شود\n\n";
                $urlMessage .= "برای لغو: /cancel یا /panel";
                
                $keyboard = [
                    'inline_keyboard' => [
                        [['text' => '❌ لغو', 'callback_data' => 'admin_cancel']]
                    ]
                ];
                
                sendMessage($chatId, $urlMessage, $keyboard);
                saveUserSession($userId, 'waiting_button_response_url', [
                    'button_text' => $buttonText,
                    'type' => 'url'
                ]);
            }
            
            return true;
            
        case 'waiting_button_response_text':
            // تکمیل افزودن دکمه با متن پاسخ
            $buttonText = $data['button_text'];
            $responseText = $text;
            
            $settings = getBotSettingsFromFile();
            $buttons = $settings['buttons'] ?? [];
            
            // اضافه کردن دکمه به یک سطر جدید
            $buttons[] = [
                [
                    'text' => $buttonText,
                    'type' => 'text',
                    'response' => $responseText
                ]
            ];
            
            $settings['buttons'] = $buttons;
            
            if (saveBotSettingsToFile($settings)) {
                $successMessage = "✅ <b>دکمه با متن پاسخ با موفقیت اضافه شد</b>\n\n";
                $successMessage .= "📝 متن دکمه: <b>" . $buttonText . "</b>\n";
                $successMessage .= "💬 متن پاسخ: <code>" . htmlspecialchars($responseText) . "</code>\n\n";
                $successMessage .= "دکمه به انتهای لیست اضافه شده است.";
                
                sendMessage($chatId, $successMessage);
            } else {
                sendMessage($chatId, "❌ خطا در ذخیره تنظیمات. لطفاً دوباره تلاش کنید.");
            }
            
            clearUserSession($userId);
            return true;
            
        case 'waiting_button_response_url':
            // تکمیل افزودن دکمه با URL
            $buttonText = $data['button_text'];
            $buttonUrl = $text;
            
            // بررسی معتبر بودن URL
            if (!filter_var($buttonUrl, FILTER_VALIDATE_URL) || 
                (strpos($buttonUrl, 'http://') !== 0 && strpos($buttonUrl, 'https://') !== 0)) {
                sendMessage($chatId, "❌ لینک نامعتبر است! لینک باید با http:// یا https:// شروع شود.\n\nلطفاً دوباره لینک را ارسال کنید:");
                return true;
            }
            
            $settings = getBotSettingsFromFile();
            $buttons = $settings['buttons'] ?? [];
            
            // اضافه کردن دکمه به یک سطر جدید
            $buttons[] = [
                [
                    'text' => $buttonText,
                    'type' => 'url',
                    'url' => $buttonUrl
                ]
            ];
            
            $settings['buttons'] = $buttons;
            
            if (saveBotSettingsToFile($settings)) {
                $successMessage = "✅ <b>دکمه با لینک با موفقیت اضافه شد</b>\n\n";
                $successMessage .= "📝 متن دکمه: <b>" . $buttonText . "</b>\n";
                $successMessage .= "🔗 لینک: <code>" . $buttonUrl . "</code>\n\n";
                $successMessage .= "دکمه به انتهای لیست اضافه شده است.";
                
                sendMessage($chatId, $successMessage);
            } else {
                sendMessage($chatId, "❌ خطا در ذخیره تنظیمات. لطفاً دوباره تلاش کنید.");
            }
            
            clearUserSession($userId);
            return true;
            
        default:
            clearUserSession($userId);
            return false;
    }
} 