<?php
/**
 * توابع اصلی ربات تلگرام اینستاگرام
 * 
 * این فایل شامل توابع اصلی مورد نیاز برای ارتباط با API تلگرام و اینستاگرام است
 */

require_once 'config.php';

/**
 * دانلود پست/ریل با Instagram120 API
 * 
 * @param string $url لینک پست/ریل
 * @return array پاسخ API
 */
function getPostFromInstagram120($url) {
    // استخراج shortcode از URL
    // مثال: https://www.instagram.com/p/ABC123/ یا /reel/ABC123/
    if (preg_match('/instagram\.com\/(p|reel|tv)\/([a-zA-Z0-9_-]+)/', $url, $matches)) {
        $shortcode = $matches[2];
    } else {
        return ['success' => false, 'error' => 'Invalid Instagram URL'];
    }
    
    // استفاده از یک مسیر API برای همه انواع محتوا
    // Instagram120 API مسیر جداگانه برای ریل ندارد
    $apiUrl = "https://" . INSTAGRAM120_HOST . "/api/instagram/media/by/shortcode";
    
    $postData = json_encode(['shortcode' => $shortcode]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); // افزایش تایم‌اوت به 60 ثانیه
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); // تنظیم زمان اتصال
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-rapidapi-host: ' . INSTAGRAM120_HOST,
        'x-rapidapi-key: ' . INSTAGRAM120_KEY
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    error_log("Instagram120 Post Request URL: " . $apiUrl);
    error_log("Instagram120 Post Request Data: " . $postData);
    error_log("Instagram120 Post Response Code: " . $httpCode);
    error_log("Instagram120 Post Response: " . substr($response, 0, 500));
    
    if ($error) {
        error_log("Instagram120 Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        return ['success' => false, 'error' => 'Invalid JSON response'];
    }
    
    // بررسی موفقیت - فرمت: {status: "success", data: {...}}
    if (isset($result['status']) && $result['status'] === 'success' && isset($result['data'])) {
        return ['success' => true, 'data' => $result];
    }
    
    return ['success' => false, 'error' => 'No data found', 'data' => $result];
}

/**
 * دانلود استوری با Fast-Creat API (با لینک مستقیم)
 * 
 * @param string $storyUrl لینک استوری یا هایلایت اینستاگرام
 * @return array پاسخ API
 */
function getStoryFromFastCreatAPI($storyUrl) {
    // پاکسازی URL استوری با حذف پارامترهای اضافی
    $cleanUrl = preg_replace('/\?.*$/', '', $storyUrl);
    error_log("Using cleaned story URL: $cleanUrl");
    
    // استفاده از تابع callInstagramAPI با type=story (به جای story2) و url کامل
    $result = callInstagramAPI('story', ['url' => $cleanUrl]);
    
    error_log("Fast-Creat Story API - Original URL: " . $storyUrl);
    error_log("Fast-Creat Story API - Response: " . json_encode($result, JSON_UNESCAPED_UNICODE));
    
    return $result;
}

/**
 * دانلود استوری با Instagram120 API
 * 
 * @param string $username نام کاربری اینستاگرام
 * @return array پاسخ API
 */
function getStoriesFromInstagram120($username) {
    $apiUrl = "https://" . INSTAGRAM120_HOST . "/api/instagram/stories";
    
    $postData = json_encode(['username' => $username]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); // افزایش تایم‌اوت به 60 ثانیه
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); // تنظیم زمان اتصال
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-rapidapi-host: ' . INSTAGRAM120_HOST,
        'x-rapidapi-key: ' . INSTAGRAM120_KEY
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    error_log("Instagram120 Story Request URL: " . $apiUrl);
    error_log("Instagram120 Story Request Data: " . $postData);
    error_log("Instagram120 Story Response Code: " . $httpCode);
    error_log("Instagram120 Story Response: " . substr($response, 0, 500));
    
    if ($error) {
        error_log("Instagram120 Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        return ['success' => false, 'error' => 'Invalid JSON response'];
    }
    
    // بررسی موفقیت - فرمت: {status: "success", data: {items: [...]}}
    if (isset($result['status']) && $result['status'] === 'success' && isset($result['data'])) {
        return ['success' => true, 'data' => $result];
    }
    
    return ['success' => false, 'error' => 'No stories found', 'data' => $result];
}

/**
 * دانلود استوری با RapidAPI
 * 
 * @param string $username نام کاربری اینستاگرام
 * @return array پاسخ API
 */
function getStoriesFromRapidAPI($username) {
    $url = "https://" . RAPIDAPI_HOST . "/v1/stories?username_or_id_or_url=" . urlencode($username);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'X-RapidAPI-Key: ' . RAPIDAPI_KEY,
        'X-RapidAPI-Host: ' . RAPIDAPI_HOST
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    // لاگ برای دیباگ
    error_log("RapidAPI Story Request URL: " . $url);
    error_log("RapidAPI Story Response Code: " . $httpCode);
    error_log("RapidAPI Story Response: " . $response);
    
    if ($error) {
        error_log("RapidAPI Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        return ['success' => false, 'error' => 'Invalid JSON response'];
    }
    
    // بررسی موفقیت
    if (isset($result['data']) && isset($result['data']['items'])) {
        return ['success' => true, 'data' => $result];
    }
    
    return ['success' => false, 'error' => 'No stories found', 'data' => $result];
}

/**
 * دانلود پست/ریل با RapidAPI
 * 
 * @param string $url لینک پست/ریل
 * @return array پاسخ API
 */
function getPostFromRapidAPI($url) {
    $apiUrl = "https://" . RAPIDAPI_HOST . "/v1.2/posts?code_or_id_or_url=" . urlencode($url);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'X-RapidAPI-Key: ' . RAPIDAPI_KEY,
        'X-RapidAPI-Host: ' . RAPIDAPI_HOST
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    error_log("RapidAPI Post Request URL: " . $apiUrl);
    error_log("RapidAPI Post Response Code: " . $httpCode);
    
    if ($error) {
        error_log("RapidAPI Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        return ['success' => false, 'error' => 'Invalid JSON response'];
    }
    
    if (isset($result['data'])) {
        return ['success' => true, 'data' => $result];
    }
    
    return ['success' => false, 'error' => 'No data found', 'data' => $result];
}

/**
 * ارسال درخواست به API اینستاگرام
 * 
 * @param string $type نوع درخواست (info, post, story, etc.)
 * @param array $params پارامترهای اضافی
 * @return array پاسخ API به صورت آرایه
 */
function callInstagramAPI($type, $params = []) {
    // انتخاب رندم یکی از لایسنس‌ها برای جلوگیری از مشکل ضد اسپم
    $apiKeys = INSTAGRAM_API_KEYS;
    $randomKey = $apiKeys[array_rand($apiKeys)];
    
    // ساخت پارامترهای درخواست
    $apiParams = [
        'apikey' => $randomKey,
        'type' => $type
    ];
    
    // افزودن پارامترهای اضافی
    $apiParams = array_merge($apiParams, $params);
    
    // ساخت URL با http_build_query
    $url = INSTAGRAM_API_URL . '?' . http_build_query($apiParams);
    
    // ثبت لاگ برای دیباگ
    error_log("API Request URL: " . $url);
    error_log("API Request Params: " . json_encode($apiParams, JSON_UNESCAPED_UNICODE));
    
    // ارسال درخواست با CURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); // افزایش تایم‌اوت به 60 ثانیه
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); // تنظیم زمان اتصال
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    curl_close($ch);
    
    // ثبت درخواست در دیتابیس
    logApiCall($type, $params, $httpCode, $error ? $error : 'success');
    
    // بررسی خطاها
    if ($error) {
        error_log("Instagram API Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    // تبدیل پاسخ JSON به آرایه
    $result = json_decode($response, true);
    
    if (!$result || isset($result['error'])) {
        $errorMsg = isset($result['error']) ? $result['error'] : 'Unknown error';
        error_log("Instagram API Error: $errorMsg");
        return ['success' => false, 'error' => $errorMsg];
    }
    
    // بررسی اینکه result null نباشه (رییل private/deleted)
    if (isset($result['result']) && $result['result'] === null) {
        error_log("Instagram API returned null result (private/deleted/unavailable)");
        return ['success' => false, 'error' => 'Content is private, deleted, or unavailable'];
    }
    
    // بررسی اینکه result وجود داشته باشه و محتوا داشته باشه
    if (isset($result['ok']) && $result['ok'] === true && isset($result['status']) && $result['status'] === 'successfully') {
        // فرمت fast-creat.ir: اگر result null است، محتوا در دسترس نیست
        if (!isset($result['result']) || empty($result['result'])) {
            error_log("Instagram API returned empty result");
            return ['success' => false, 'error' => 'No content available'];
        }
    }
    
    return ['success' => true, 'data' => $result];
}

/**
 * ثبت درخواست API در دیتابیس
 * 
 * @param string $type نوع درخواست
 * @param array $params پارامترها
 * @param int $httpCode کد پاسخ HTTP
 * @param string $status وضعیت درخواست
 */
function logApiCall($type, $params, $httpCode, $status) {
    $conn = getDbConnection();
    
    $telegramId = isset($params['telegram_id']) ? $params['telegram_id'] : (isset($params['user_id']) ? $params['user_id'] : 0);
    $paramsJson = json_encode($params, JSON_UNESCAPED_UNICODE);
    
    $stmt = $conn->prepare("INSERT INTO api_calls (telegram_id, endpoint, params, response_code, is_error, error_message, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    
    if ($stmt === false) {
        error_log("Database prepare error in logApiCall: " . $conn->error);
        $conn->close();
        return;
    }
    
    $isError = ($status !== 'success') ? 1 : 0;
    $errorMessage = ($status !== 'success') ? $status : '';
    
    $stmt->bind_param("ississ", $telegramId, $type, $paramsJson, $httpCode, $isError, $errorMessage);
    $stmt->execute();
    $stmt->close();
    
    $conn->close();
}

/**
 * مدیریت آمار و اطلاعات کاربر
 * 
 * @param int $telegramId شناسه تلگرام کاربر
 * @param string $action نوع عملیات (view, download, etc.)
 * @param string $contentType نوع محتوا (post, story, profile, etc.)
 * @return bool نتیجه عملیات
 */
function handleStats($telegramId, $action = 'view', $contentType = 'unknown') {
    $conn = getDbConnection();
    
    // افزایش تعداد درخواست‌های کاربر
    $stmt = $conn->prepare("UPDATE users SET request_count = request_count + 1, last_activity = NOW() WHERE telegram_id = ?");
    
    if ($stmt === false) {
        error_log("Database prepare error in handleStats: " . $conn->error);
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("i", $telegramId);
    $result = $stmt->execute();
    $stmt->close();
    
    // ثبت آمار استفاده در جدول api_calls
    $params = [
        'telegram_id' => $telegramId,
        'action' => $action,
        'content_type' => $contentType
    ];
    
    logApiCall('stats', $params, 200, 'success');
    
    $conn->close();
    return $result;
}

/**
 * ارسال پیام متنی به کاربر تلگرام
 * 
 * @param int $chatId شناسه چت
 * @param string $text متن پیام
 * @param array $keyboard دکمه‌های کیبورد (اختیاری)
 * @return array نتیجه ارسال پیام
 */
function sendMessage($chatId, $text, $keyboard = null) {
    $data = [
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ];
    
    if ($keyboard !== null) {
        $data['reply_markup'] = json_encode($keyboard);
    }
    
    return sendTelegramRequest('sendMessage', $data);
}

/**
 * ارسال عکس به کاربر تلگرام
 * 
 * @param int $chatId شناسه چت
 * @param string $photo آدرس یا فایل عکس
 * @param string $caption توضیحات عکس (اختیاری)
 * @param array $keyboard دکمه‌های کیبورد (اختیاری)
 * @return array نتیجه ارسال عکس
 */
function sendPhoto($chatId, $photo, $caption = '', $keyboard = null) {
    $data = [
        'chat_id' => $chatId,
        'photo' => $photo,
        'parse_mode' => 'HTML',
    ];
    
    if (!empty($caption)) {
        $data['caption'] = $caption;
    }
    
    if ($keyboard !== null) {
        $data['reply_markup'] = json_encode($keyboard);
    }
    
    $result = sendTelegramRequest('sendPhoto', $data);
    
    // اگر ارسال عکس موفق نبود، لینک دانلود مستقیم بفرست
    if (!$result['success'] && strpos($photo, 'http') === 0) {
        $errorCaption = $caption . "\n\n⚠️ تلگرام نتوانست عکس را دانلود کند.\n\n";
        $errorCaption .= "🔗 <b>لینک دانلود مستقیم:</b>\n";
        $errorCaption .= "<a href='" . $photo . "'>کلیک کنید برای دانلود</a>";
        
        $downloadKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬇️ دانلود مستقیم', 'url' => $photo]
                ]
            ]
        ];
        
        return sendMessage($chatId, $errorCaption, $downloadKeyboard);
    }
    
    return $result;
}

/**
 * ارسال ویدیو به کاربر تلگرام
 * 
 * @param int $chatId شناسه چت
 * @param string $video آدرس یا فایل ویدیو
 * @param string $caption توضیحات ویدیو (اختیاری)
 * @param array $keyboard دکمه‌های کیبورد (اختیاری)
 * @return array نتیجه ارسال ویدیو
 */
function sendVideo($chatId, $video, $caption = '', $keyboard = null) {
    $data = [
        'chat_id' => $chatId,
        'video' => $video,
        'parse_mode' => 'HTML',
    ];
    
    if (!empty($caption)) {
        $data['caption'] = $caption;
    }
    
    if ($keyboard !== null) {
        $data['reply_markup'] = json_encode($keyboard);
    }
    
    $result = sendTelegramRequest('sendVideo', $data);
    
    // اگر ارسال ویدیو موفق نبود، لینک دانلود مستقیم بفرست
    if (!$result['success'] && strpos($video, 'http') === 0) {
        $errorCaption = $caption . "\n\n⚠️ تلگرام نتوانست ویدیو را دانلود کند.\n\n";
        $errorCaption .= "🔗 <b>لینک دانلود مستقیم:</b>\n";
        $errorCaption .= "<a href='" . $video . "'>کلیک کنید برای دانلود</a>";
        
        $downloadKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬇️ دانلود مستقیم', 'url' => $video]
                ]
            ]
        ];
        
        return sendMessage($chatId, $errorCaption, $downloadKeyboard);
    }
    
    return $result;
}

/**
 * فوروارد پیام به کاربر تلگرام
 * 
 * @param int $chatId شناسه چت مقصد
 * @param int $fromChatId شناسه چت مبدا
 * @param int $messageId شناسه پیام
 * @return array نتیجه فوروارد پیام
 */
function forwardMessage($chatId, $fromChatId, $messageId) {
    $data = [
        'chat_id' => $chatId,
        'from_chat_id' => $fromChatId,
        'message_id' => $messageId
    ];
    
    return sendTelegramRequest('forwardMessage', $data);
}

/**
 * ارسال درخواست به API تلگرام
 * 
 * @param string $method نام متد API
 * @param array $data داده‌های درخواست
 * @return array نتیجه درخواست
 */
function sendTelegramRequest($method, $data) {
    $url = TELEGRAM_API_URL . '/' . $method;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    if ($error) {
        error_log("Telegram API Error: $error");
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if (!$result || !$result['ok']) {
        $errorMsg = isset($result['description']) ? $result['description'] : 'Unknown error';
        error_log("Telegram API Error: $errorMsg");
        return ['success' => false, 'error' => $errorMsg];
    }
    
    return ['success' => true, 'data' => $result['result']];
}

/**
 * ایجاد کیبورد اصلی ربات (از فایل JSON)
 * 
 * @return array آرایه کیبورد
 */
function getMainKeyboard() {
    $settings = getBotSettingsFromFile();
    
    if (empty($settings['buttons'])) {
        // فالبک اگر فایل خراب باشد
        return [
            'keyboard' => [
                [
                    ['text' => '📥 دانلود پست/ریل'],
                    ['text' => '📹 دانلود استوری']
                ],
                [
                    ['text' => '❓ راهنما']
                ],
                [
                    ['text' => 'سفارش ربات | خرید VPN']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
    }
    
    $keyboard = [];
    
    foreach ($settings['buttons'] as $row) {
        $keyboardRow = [];
        foreach ($row as $button) {
            // فقط دکمه‌های معمولی (بدون URL) رو به کیبورد اضافه کن
            if (!isset($button['type']) || $button['type'] !== 'url') {
                $keyboardRow[] = ['text' => $button['text']];
            }
        }
        if (!empty($keyboardRow)) {
            $keyboard[] = $keyboardRow;
        }
    }
    
    return [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}

/**
 * ایجاد کیبورد inline برای دکمه‌های URL
 * 
 * @return array|null آرایه کیبورد یا null
 */
function getUrlButtonsKeyboard() {
    $settings = getBotSettingsFromFile();
    
    if (empty($settings['buttons'])) {
        return null;
    }
    
    $inlineKeyboard = [];
    
    foreach ($settings['buttons'] as $row) {
        $inlineRow = [];
        foreach ($row as $button) {
            // فقط دکمه‌های URL رو به inline keyboard اضافه کن
            if (isset($button['type']) && $button['type'] === 'url' && isset($button['url'])) {
                $inlineRow[] = [
                    'text' => $button['text'],
                    'url' => $button['url']
                ];
            }
        }
        if (!empty($inlineRow)) {
            $inlineKeyboard[] = $inlineRow;
        }
    }
    
    if (empty($inlineKeyboard)) {
        return null;
    }
    
    return [
        'inline_keyboard' => $inlineKeyboard
    ];
}

/**
 * دریافت تنظیمات ربات از فایل JSON
 * 
 * @return array تنظیمات ربات
 */
function getBotSettingsFromFile() {
    $settingsFile = __DIR__ . '/bot_settings.json';
    
    if (!file_exists($settingsFile)) {
        // ایجاد فایل با تنظیمات پیش‌فرض
        $defaultSettings = [
            'start_message' => "👋 سلام <b>{first_name}</b>!\n\nبه ربات دانلود اینستاگرام خوش آمدید. 🚀\n\nبا این ربات می‌توانید:\n\n📥 <b>دانلود پست و ریل</b>\n• فقط لینک پست یا ریل را ارسال کنید\n\n📹 <b>دانلود استوری</b>\n• لینک استوری یا نام کاربری را ارسال کنید\n\n💡 <b>نکته:</b> فقط کافیست لینک اینستاگرام را بفرستید!\nربات به طور خودکار محتوا را دانلود می‌کند.\n\n━━━━━━━━━━━━━━━\n👥 <b>{total_users}</b> کاربر فعال\n📥 <b>{total_downloads}</b> دانلود موفق",
            'buttons' => [
                [
                    ['text' => '📥 دانلود پست/ریل'],
                    ['text' => '📹 دانلود استوری']
                ],
                [
                    ['text' => '❓ راهنما']
                ],
                [
                    ['text' => 'سفارش ربات | خرید VPN']
                ]
            ]
        ];
        
        file_put_contents($settingsFile, json_encode($defaultSettings, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
        return $defaultSettings;
    }
    
    $settings = json_decode(file_get_contents($settingsFile), true);
    
    if (!$settings) {
        // اگر فایل خراب بود، تنظیمات پیش‌فرض برگردان
        return [
            'start_message' => "👋 سلام <b>{first_name}</b>!\n\nبه ربات دانلود اینستاگرام خوش آمدید. 🚀",
            'buttons' => [
                [
                    ['text' => '📥 دانلود پست/ریل'],
                    ['text' => '📹 دانلود استوری']
                ],
                [
                    ['text' => '❓ راهنما']
                ]
            ]
        ];
    }
    
    return $settings;
}

/**
 * ذخیره تنظیمات ربات در فایل JSON
 * 
 * @param array $settings تنظیمات ربات
 * @return bool نتیجه عملیات
 */
function saveBotSettingsToFile($settings) {
    $settingsFile = __DIR__ . '/bot_settings.json';
    
    $result = file_put_contents($settingsFile, json_encode($settings, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    
    return $result !== false;
}

/**
 * بررسی محدودیت نرخ درخواست کاربر
 * 
 * @param int $userId شناسه کاربر
 * @return bool آیا کاربر محدودیت دارد
 */
function checkRateLimit($telegramId) {
    $conn = getDbConnection();
    
    // بررسی تعداد درخواست‌های کاربر در بازه زمانی مشخص
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM api_calls WHERE telegram_id = ? AND created_at > NOW() - INTERVAL ? SECOND");
    
    // بررسی خطا در prepare
    if ($stmt === false) {
        error_log("Database prepare error: " . $conn->error);
        $conn->close();
        return false;
    }
    
    // ایجاد متغیر محلی برای استفاده به عنوان پارامتر ارجاعی
    $period = RATE_LIMIT_PERIOD;
    
    $stmt->bind_param("ii", $telegramId, $period);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $count = $row['count'];
    
    $stmt->close();
    $conn->close();
    
    // اگر تعداد درخواست‌ها بیشتر از حد مجاز باشد
    return $count >= RATE_LIMIT;
}

/**
 * بررسی عضویت کاربر در کانال‌های اجباری
 * 
 * @param int $userId شناسه کاربر
 * @param bool $forceJoin آیا پیام فورس جوین نمایش داده شود
 * @return array نتیجه بررسی (وضعیت و لیست کانال‌های غیرعضو)
 */
function checkRequiredChannels($userId, $forceJoin = false) {
    $conn = getDbConnection();
    
    // دریافت لیست کانال‌های اجباری
    $stmt = $conn->prepare("SELECT channel_id, channel_username, channel_name FROM channels_required");
    $stmt->execute();
    $result = $stmt->get_result();
    
    $notJoinedChannels = [];
    $allJoined = true;
    
    while ($row = $result->fetch_assoc()) {
        $channelUsername = $row['channel_username'];
        $channelName = $row['channel_name'];
        $channelId = $row['channel_id'];
        
        // بررسی عضویت کاربر در کانال
        // برای کانال‌های خصوصی از ID استفاده می‌کنیم
        $chatIdentifier = !empty($channelId) ? $channelId : $channelUsername;
        $chatMember = getChatMember($chatIdentifier, $userId);
        
        if (!$chatMember['success'] || 
            !in_array($chatMember['data']['status'], ['creator', 'administrator', 'member'])) {
            $notJoinedChannels[] = [
                'username' => $channelUsername,
                'name' => $channelName,
                'id' => $channelId
            ];
            $allJoined = false;
            
            // اگر فورس جوین فعال باشد، پیام عضویت اجباری را ارسال کنیم
            if ($forceJoin) {
                sendForceJoinMessage($userId, $notJoinedChannels);
                break; // بعد از ارسال پیام فورس جوین، دیگر نیازی به بررسی بقیه کانال‌ها نیست
            }
        }
    }
    
    $stmt->close();
    $conn->close();
    
    return [
        'joined' => $allJoined,
        'not_joined' => $notJoinedChannels
    ];
}

/**
 * ارسال پیام فورس جوین به کاربر
 * 
 * @param int $userId شناسه کاربر
 * @param array $notJoinedChannels لیست کانال‌های غیرعضو
 * @return void
 */
function sendForceJoinMessage($userId, $notJoinedChannels) {
    // دریافت اطلاعات کامل کانال‌ها از دیتابیس
    $conn = getDbConnection();
    $channelIds = [];
    foreach ($notJoinedChannels as $channel) {
        $channelIds[] = "'" . $channel['username'] . "'";
    }
    
    $channelsInfo = [];
    if (!empty($channelIds)) {
        $query = "SELECT channel_username, channel_name, invite_link FROM channels_required WHERE channel_username IN (" . implode(',', $channelIds) . ")";
        $result = $conn->query($query);
        
        while ($row = $result->fetch_assoc()) {
            $channelsInfo[$row['channel_username']] = $row;
        }
    }
    $conn->close();
    
    // ساخت پیام فورس جوین
    $message = "⚠️ <b>برای استفاده از ربات باید عضو کانال(های) زیر باشید:</b>\n\n";
    
    $inlineKeyboard = ['inline_keyboard' => []];
    
    foreach ($notJoinedChannels as $channel) {
        $channelUsername = ltrim($channel['username'], '@');
        $channelInfo = isset($channelsInfo[$channelUsername]) ? $channelsInfo[$channelUsername] : null;
        $channelName = $channelInfo ? $channelInfo['channel_name'] : $channel['name'];
        $inviteLink = $channelInfo && !empty($channelInfo['invite_link']) ? $channelInfo['invite_link'] : 'https://t.me/' . $channelUsername;
        
        $message .= "• " . $channelName . "\n";
        
        // اضافه کردن دکمه برای هر کانال
        $inlineKeyboard['inline_keyboard'][] = [
            [
                'text' => '🔗 عضویت در ' . $channelName,
                'url' => $inviteLink
            ]
        ];
    }
    
    // اضافه کردن دکمه بررسی مجدد
    $inlineKeyboard['inline_keyboard'][] = [
        [
            'text' => '✅ بررسی عضویت من',
            'callback_data' => 'check_join'
        ]
    ];
    
    $message .= "\n🔄 پس از عضویت، روی دکمه «بررسی عضویت من» کلیک کنید.";
    
    // ارسال پیام با دکمه‌های اینلاین
    sendMessage($userId, $message, $inlineKeyboard);
}

/**
 * دریافت اطلاعات عضویت کاربر در چت
 * 
 * @param string $chatId شناسه یا نام کاربری چت
 * @param int $userId شناسه کاربر
 * @return array نتیجه درخواست
 */
function getChatMember($chatId, $userId) {
    $data = [
        'chat_id' => $chatId,
        'user_id' => $userId
    ];
    
    return sendTelegramRequest('getChatMember', $data);
}

/**
 * ثبت یا به‌روزرسانی کاربر در دیتابیس
 * 
 * @param array $user اطلاعات کاربر تلگرام
 * @return int شناسه کاربر در دیتابیس
 */
function saveUser($user) {
    $conn = getDbConnection();
    
    $telegramId = $user['id'];
    $username = isset($user['username']) ? $user['username'] : '';
    $firstName = isset($user['first_name']) ? $user['first_name'] : '';
    $lastName = isset($user['last_name']) ? $user['last_name'] : '';
    
    // بررسی وجود کاربر
    $stmt = $conn->prepare("SELECT id FROM users WHERE telegram_id = ?");
    $stmt->bind_param("i", $telegramId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        // به‌روزرسانی کاربر موجود
        $row = $result->fetch_assoc();
        $userId = $row['id'];
        
        $stmt = $conn->prepare("UPDATE users SET username = ?, first_name = ?, last_name = ?, last_activity = NOW() WHERE id = ?");
        $stmt->bind_param("sssi", $username, $firstName, $lastName, $userId);
        $stmt->execute();
    } else {
        // ایجاد کاربر جدید
        $isAdmin = isAdmin($telegramId) ? 1 : 0;
        
        $stmt = $conn->prepare("INSERT INTO users (telegram_id, username, first_name, last_name, is_admin, joined_at, last_activity) VALUES (?, ?, ?, ?, ?, NOW(), NOW())");
        $stmt->bind_param("isssi", $telegramId, $username, $firstName, $lastName, $isAdmin);
        $stmt->execute();
        
        $userId = $conn->insert_id;
    }
    
    $stmt->close();
    $conn->close();
    
    return $userId;
}

/**
 * اعتبارسنجی نام کاربری اینستاگرام
 * 
 * @param string $username نام کاربری
 * @return bool آیا نام کاربری معتبر است
 */
function validateInstagramUsername($username) {
    return preg_match('/^[a-zA-Z0-9._]{1,30}$/', $username);
}

/**
 * اعتبارسنجی URL اینستاگرام
 * 
 * @param string $url آدرس URL
 * @return bool آیا URL معتبر است
 */
function validateInstagramUrl($url) {
    return preg_match('/^https?:\/\/(www\.)?instagram\.com\/(p|reel|tv)\/[a-zA-Z0-9_-]+\/?/', $url);
}

/**
 * ارسال پیام خطا به کاربر
 * 
 * @param int $chatId شناسه چت
 * @param string $errorType نوع خطا
 * @return array نتیجه ارسال پیام
 */
function sendErrorMessage($chatId, $errorType) {
    global $messages;
    
    $errorMessage = isset($messages[$errorType]) ? $messages[$errorType] : $messages['error'];
    
    return sendMessage($chatId, $errorMessage);
}

/**
 * ارسال محتوای پست اینستاگرام به کاربر
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 * @param array $postData اطلاعات پست
 * @return array نتیجه ارسال پیام
 */
function sendPostContent($chatId, $userId, $postData) {
    // بررسی داده‌های دریافتی
    if (empty($postData) || !is_array($postData)) {
        error_log("Invalid post data: empty or not an array");
        return ['success' => false, 'error' => 'Invalid post data'];
    }
    
    // بررسی اینکه result null نباشه (محتوا در دسترس نیست)
    if (isset($postData['result']) && $postData['result'] === null) {
        error_log("Post data has null result - content unavailable");
        return ['success' => false, 'error' => 'Content unavailable'];
    }
    
    // ثبت داده‌های دریافتی برای اشکال‌زدایی
    error_log("Post data received: " . json_encode($postData, JSON_UNESCAPED_UNICODE));
    
    // استخراج داده اصلی از فرمت‌های مختلف
    $actualData = $postData;
    
    // فرمت Instagram120: {status: "success", data: {...}}
    if (isset($postData['status']) && $postData['status'] === 'success' && isset($postData['data']) && !isset($postData['data']['items'])) {
        error_log("Detected Instagram120 format");
        $item = $postData['data'];
        
        // تبدیل فرمت Instagram120 به فرمت استاندارد
        $actualData = [];
        
        // استخراج کپشن
        if (isset($item['caption'])) {
            $actualData['caption'] = $item['caption'];
        }
        
        // استخراج نام کاربری
        if (isset($item['owner']) && isset($item['owner']['username'])) {
            $actualData['username'] = $item['owner']['username'];
        }
        
        // استخراج آمار
        if (isset($item['like_count'])) {
            $actualData['likes'] = $item['like_count'];
        }
        if (isset($item['comment_count'])) {
            $actualData['comments'] = $item['comment_count'];
        }
        
        // تشخیص نوع مدیا و URL
        if (isset($item['is_video']) && $item['is_video'] === true) {
            $actualData['is_video'] = true;
            if (isset($item['video_url'])) {
                $actualData['video_url'] = $item['video_url'];
            }
        } else {
            $actualData['is_video'] = false;
            if (isset($item['display_url'])) {
                $actualData['image_url'] = $item['display_url'];
            }
        }
        
        // بررسی آلبوم (carousel)
        if (isset($item['carousel_media']) && is_array($item['carousel_media']) && count($item['carousel_media']) > 0) {
            $actualData['is_album'] = true;
            $actualData['carousel'] = [];
            
            foreach ($item['carousel_media'] as $carouselItem) {
                $mediaItem = [];
                
                if (isset($carouselItem['is_video']) && $carouselItem['is_video'] === true) {
                    $mediaItem['media_type'] = 'video';
                    $mediaItem['video_url'] = $carouselItem['video_url'];
                } else {
                    $mediaItem['media_type'] = 'image';
                    $mediaItem['image_url'] = $carouselItem['display_url'];
                }
                
                if (!empty($mediaItem)) {
                    $actualData['carousel'][] = $mediaItem;
                }
            }
        }
        
        error_log("Converted Instagram120 data: " . json_encode($actualData, JSON_UNESCAPED_UNICODE));
    }
    // فرمت RapidAPI: {data: {items: [...]}}
    elseif (isset($postData['data']) && isset($postData['data']['items']) && is_array($postData['data']['items']) && !empty($postData['data']['items'])) {
        error_log("Detected RapidAPI format");
        $item = $postData['data']['items'][0];
        
        // تبدیل فرمت RapidAPI به فرمت استاندارد
        $actualData = [];
        
        // استخراج کپشن
        if (isset($item['caption']) && isset($item['caption']['text'])) {
            $actualData['caption'] = $item['caption']['text'];
        }
        
        // استخراج نام کاربری
        if (isset($item['user']) && isset($item['user']['username'])) {
            $actualData['username'] = $item['user']['username'];
        }
        
        // استخراج آمار
        if (isset($item['like_count'])) {
            $actualData['likes'] = $item['like_count'];
        }
        if (isset($item['comment_count'])) {
            $actualData['comments'] = $item['comment_count'];
        }
        
        // تشخیص نوع مدیا و URL
        if (isset($item['video_versions']) && !empty($item['video_versions'])) {
            $actualData['is_video'] = true;
            $actualData['video_url'] = $item['video_versions'][0]['url'];
        } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
            $actualData['is_video'] = false;
            $actualData['image_url'] = $item['image_versions2']['candidates'][0]['url'];
        }
        
        // بررسی آلبوم (carousel)
        if (isset($item['carousel_media']) && is_array($item['carousel_media']) && count($item['carousel_media']) > 1) {
            $actualData['is_album'] = true;
            $actualData['carousel'] = [];
            
            foreach ($item['carousel_media'] as $carouselItem) {
                $mediaItem = [];
                
                if (isset($carouselItem['video_versions']) && !empty($carouselItem['video_versions'])) {
                    $mediaItem['media_type'] = 'video';
                    $mediaItem['video_url'] = $carouselItem['video_versions'][0]['url'];
                } elseif (isset($carouselItem['image_versions2']) && isset($carouselItem['image_versions2']['candidates'])) {
                    $mediaItem['media_type'] = 'image';
                    $mediaItem['image_url'] = $carouselItem['image_versions2']['candidates'][0]['url'];
                }
                
                if (!empty($mediaItem)) {
                    $actualData['carousel'][] = $mediaItem;
                }
            }
        }
        
        error_log("Converted RapidAPI data: " . json_encode($actualData, JSON_UNESCAPED_UNICODE));
    }
    
    // فرمت جدید API: {ok: true, result: {status: "success", result: [...]}}
    if (isset($postData['ok']) && $postData['ok'] === true && isset($postData['result']['result'])) {
        if (is_array($postData['result']['result']) && !empty($postData['result']['result'])) {
            $allItems = $postData['result']['result'];
            $actualData = $allItems[0]; // اولین آیتم = اطلاعات اصلی
            
            // بررسی آلبوم: اگر بیش از 1 آیتم داشت و اولی is_album بود
            if (count($allItems) > 1 && isset($actualData['is_album']) && $actualData['is_album'] === true) {
                // پردازش آلبوم
                $caption = isset($actualData['caption']) ? $actualData['caption'] : '';
                $username = isset($actualData['username']) ? $actualData['username'] : '';
                
                $albumMessage = "📥 <b>دانلود از اینستاگرام</b>\n\n";
                if (!empty($username)) {
                    $albumMessage .= "👤 <b>کاربر:</b> @{$username}\n";
                }
                $albumMessage .= "📁 <b>آلبوم:</b> " . (count($allItems) - 1) . " رسانه\n";
                
                if (!empty($caption)) {
                    $maxCaptionLength = 120;
                    $shortCaption = mb_strlen($caption) > $maxCaptionLength 
                        ? mb_substr($caption, 0, $maxCaptionLength) . "..." 
                        : $caption;
                    $albumMessage .= "\n<b>کپشن:</b>\n{$shortCaption}";
                }
                
                $albumMessage .= "\n\n📥 دانلود شده توسط: " . BOT_USERNAME;
                
                sendMessage($chatId, $albumMessage);
                
                // ارسال تمام رسانه‌ها (به جز آیتم اول که info بود)
                for ($i = 1; $i < count($allItems); $i++) {
                    $item = $allItems[$i];
                    $itemCaption = "🖼 رسانه " . $i . " از " . (count($allItems) - 1) . "\n\n📥 " . BOT_USERNAME;
                    
                    if (isset($item['is_video']) && $item['is_video'] && isset($item['video_url'])) {
                        sendVideo($chatId, $item['video_url'], $itemCaption);
                    } elseif (isset($item['image_url'])) {
                        sendPhoto($chatId, $item['image_url'], $itemCaption);
                    }
                    
                    usleep(400000); // 400ms تاخیر برای جلوگیری از flood
                }
                
                return ['success' => true];
            }
        } elseif (isset($postData['result'])) {
            $actualData = $postData['result'];
        }
    } elseif (isset($postData['status']) && $postData['status'] === 'successfully' && isset($postData['result']['result'])) {
        if (is_array($postData['result']['result']) && !empty($postData['result']['result'])) {
            $allItems = $postData['result']['result'];
            $actualData = $allItems[0];
            
            // همان پردازش آلبوم
            if (count($allItems) > 1 && isset($actualData['is_album']) && $actualData['is_album'] === true) {
                $caption = isset($actualData['caption']) ? $actualData['caption'] : '';
                $username = isset($actualData['username']) ? $actualData['username'] : '';
                
                $albumMessage = "📥 <b>دانلود از اینستاگرام</b>\n\n";
                if (!empty($username)) {
                    $albumMessage .= "👤 <b>کاربر:</b> @{$username}\n";
                }
                $albumMessage .= "📁 <b>آلبوم:</b> " . (count($allItems) - 1) . " رسانه\n";
                
                if (!empty($caption)) {
                    $maxCaptionLength = 120;
                    $shortCaption = mb_strlen($caption) > $maxCaptionLength 
                        ? mb_substr($caption, 0, $maxCaptionLength) . "..." 
                        : $caption;
                    $albumMessage .= "\n<b>کپشن:</b>\n{$shortCaption}";
                }
                
                $albumMessage .= "\n\n📥 دانلود شده توسط: " . BOT_USERNAME;
                
                sendMessage($chatId, $albumMessage);
                
                for ($i = 1; $i < count($allItems); $i++) {
                    $item = $allItems[$i];
                    $itemCaption = "🖼 رسانه " . $i . " از " . (count($allItems) - 1) . "\n\n📥 " . BOT_USERNAME;
                    
                    if (isset($item['is_video']) && $item['is_video'] && isset($item['video_url'])) {
                        sendVideo($chatId, $item['video_url'], $itemCaption);
                    } elseif (isset($item['image_url'])) {
                        sendPhoto($chatId, $item['image_url'], $itemCaption);
                    }
                    
                    usleep(400000);
                }
                
                return ['success' => true];
            }
        }
    } elseif (isset($postData['status']) && $postData['status'] === 'ok' && isset($postData['result'])) {
        $actualData = $postData['result'];
    } elseif (isset($postData['result']) && $postData['result'] === 'ok' && isset($postData['data'])) {
        $actualData = $postData['data'];
    } elseif (isset($postData['post'])) {
        $actualData = $postData['post'];
    }
    
    // لاگ داده پردازش شده
    error_log("Actual data to process: " . json_encode($actualData, JSON_UNESCAPED_UNICODE));
    
    // استخراج اطلاعات پست
    $caption = isset($actualData['caption']) ? $actualData['caption'] : (isset($actualData['caption_text']) ? $actualData['caption_text'] : '');
    $username = isset($actualData['username']) ? $actualData['username'] : (isset($actualData['owner']['username']) ? $actualData['owner']['username'] : '');
    $likes = isset($actualData['likes']) ? number_format($actualData['likes']) : (isset($actualData['like_count']) ? number_format($actualData['like_count']) : '0');
    $comments = isset($actualData['comments']) ? number_format($actualData['comments']) : (isset($actualData['comment_count']) ? number_format($actualData['comment_count']) : '0');
    
    // ساخت پیام اطلاعات پست
    $message = "📥 <b>دانلود از اینستاگرام</b>\n\n";
    
    if (!empty($username)) {
        $message .= "👤 <b>کاربر:</b> @{$username}\n";
    }
    
    if (!empty($likes) && $likes != '0') {
        $message .= "❤️ <b>لایک‌ها:</b> {$likes}\n";
    }
    
    if (!empty($comments) && $comments != '0') {
        $message .= "💬 <b>نظرات:</b> {$comments}\n";
    }
    
    if (!empty($caption)) {
        // محدود کردن طول کپشن
        $maxCaptionLength = 180;
        $shortCaption = mb_strlen($caption) > $maxCaptionLength 
            ? mb_substr($caption, 0, $maxCaptionLength) . "..." 
            : $caption;
        
        $message .= "\n<b>کپشن:</b>\n{$shortCaption}\n";
    }
    
    // اضافه کردن ID ربات
    $message .= "\n\n📥 دانلود شده توسط: " . BOT_USERNAME;
    
    // بررسی ساختارهای مختلف داده برای ریل‌ها و پست‌ها
    
    // حالت 0: ساختار API fast-creat.ir با is_video
    if (isset($actualData['is_video']) && $actualData['is_video'] === true) {
        if (isset($actualData['video_url'])) {
            return sendVideo($chatId, $actualData['video_url'], $message);
        }
    } elseif (isset($actualData['is_video']) && $actualData['is_video'] === false) {
        // عکس
        if (isset($actualData['video_img'])) {
            return sendPhoto($chatId, $actualData['video_img'], $message);
        } elseif (isset($actualData['image_url'])) {
            return sendPhoto($chatId, $actualData['image_url'], $message);
        }
    }
    

    
    // حالت 1: ساختار استاندارد با media_type
    if (isset($actualData['media_type'])) {
        if ($actualData['media_type'] === 'video' || $actualData['media_type'] === 'Video') {
            // ارسال ویدیو
            if (isset($actualData['video_url'])) {
                return sendVideo($chatId, $actualData['video_url'], $message);
            } elseif (isset($actualData['video'])) {
                return sendVideo($chatId, $actualData['video'], $message);
            }
        } elseif ($actualData['media_type'] === 'image' || $actualData['media_type'] === 'Photo') {
            // ارسال تصویر
            if (isset($actualData['image_url'])) {
                return sendPhoto($chatId, $actualData['image_url'], $message);
            } elseif (isset($actualData['image'])) {
                return sendPhoto($chatId, $actualData['image'], $message);
            } elseif (isset($actualData['display_url'])) {
                return sendPhoto($chatId, $actualData['display_url'], $message);
            }
        } elseif ($actualData['media_type'] === 'carousel' || $actualData['media_type'] === 'Album') {
            // ارسال آلبوم (چند رسانه‌ای)
            if (isset($actualData['carousel']) && is_array($actualData['carousel'])) {
                // ارسال پیام اطلاعات
                sendMessage($chatId, $message);
                
                // ارسال هر آیتم از آلبوم
                foreach ($actualData['carousel'] as $item) {
                    if (isset($item['media_type']) && ($item['media_type'] === 'video' || $item['media_type'] === 'Video') && isset($item['video_url'])) {
                        sendVideo($chatId, $item['video_url'], "");
                    } elseif (isset($item['media_type']) && ($item['media_type'] === 'image' || $item['media_type'] === 'Photo') && isset($item['image_url'])) {
                        sendPhoto($chatId, $item['image_url'], "");
                    }
                }
                
                return ['success' => true];
            }
        }
    }
    
    // حالت 2: ساختار ریل با url مستقیم
    if (isset($actualData['url']) && (strpos($actualData['url'], '.mp4') !== false || strpos($actualData['url'], 'video') !== false)) {
        return sendVideo($chatId, $actualData['url'], $message);
    }
    
    // حالت 3: ساختار ریل با video_versions
    if (isset($actualData['video_versions']) && is_array($actualData['video_versions']) && !empty($actualData['video_versions'])) {
        $videoUrl = isset($actualData['video_versions'][0]['url']) ? $actualData['video_versions'][0]['url'] : null;
        if ($videoUrl) {
            return sendVideo($chatId, $videoUrl, $message);
        }
    }
    
    // حالت 4: ساختار ریل با items
    if (isset($actualData['items']) && is_array($actualData['items']) && !empty($actualData['items'])) {
        $item = $actualData['items'][0];
        
        // بررسی نوع محتوا در آیتم
        if (isset($item['media_type']) && $item['media_type'] === 'video') {
            if (isset($item['video_url'])) {
                return sendVideo($chatId, $item['video_url'], $message);
            } elseif (isset($item['video_versions']) && is_array($item['video_versions']) && !empty($item['video_versions'])) {
                $videoUrl = $item['video_versions'][0]['url'];
                return sendVideo($chatId, $videoUrl, $message);
            }
        } elseif (isset($item['media_type']) && $item['media_type'] === 'image') {
            if (isset($item['image_url'])) {
                return sendPhoto($chatId, $item['image_url'], $message);
            } elseif (isset($item['image_versions2']) && isset($item['image_versions2']['candidates'])) {
                $imageUrl = $item['image_versions2']['candidates'][0]['url'];
                return sendPhoto($chatId, $imageUrl, $message);
            }
        }
    }
    
    // اگر هیچ محتوایی یافت نشد
    error_log("No downloadable content found in post data");
    return ['success' => false, 'error' => 'No downloadable content found'];
}

/**
 * ارسال محتوای استوری اینستاگرام به کاربر
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 * @param array $storyData اطلاعات استوری
 * @return array نتیجه ارسال پیام
 */
function sendStoryContent($chatId, $userId, $storyData) {
    // بررسی داده‌های دریافتی
    if (empty($storyData) || !is_array($storyData)) {
        return sendMessage($chatId, "❌ <b>خطا در پردازش داده‌ها</b>\n\nاطلاعات استوری قابل پردازش نیست.");
    }
    
    // استخراج اطلاعات استوری
    $username = isset($storyData['username']) ? $storyData['username'] : '';
    
    // ساخت پیام اطلاعات استوری
    $message = "📥 <b>دانلود استوری اینستاگرام</b>\n\n";
    
    if (!empty($username)) {
        $message .= "👤 <b>کاربر:</b> @{$username}\n";
    }
    
    // بررسی نوع محتوا و ارسال آن
    if (isset($storyData['media_type']) && $storyData['media_type'] === 'video') {
        // ارسال ویدیو
        if (isset($storyData['video_url'])) {
            return sendVideo($chatId, $storyData['video_url'], $message);
        }
    } elseif (isset($storyData['media_type']) && $storyData['media_type'] === 'image') {
        // ارسال تصویر
        if (isset($storyData['image_url'])) {
            return sendPhoto($chatId, $storyData['image_url'], $message);
        }
    }
    
    // اگر هیچ محتوایی یافت نشد
    return sendMessage($chatId, $message . "\n⚠️ محتوای قابل دانلود یافت نشد.");
}

/**
 * دریافت آمار کاربران
 * 
 * @return array آمار کاربران
 */
function getUserStats() {
    $conn = getDbConnection();
    
    // تعداد کل کاربران
    $totalUsers = $conn->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'];
    
    // کاربران فعال امروز
    $today = date('Y-m-d');
    $activeToday = $conn->query("SELECT COUNT(DISTINCT telegram_id) as count FROM api_calls WHERE DATE(created_at) = '$today'")->fetch_assoc()['count'];
    
    // تعداد دانلودهای امروز (فقط امروز)
    $downloadsToday = $conn->query("SELECT COUNT(*) as count FROM api_calls WHERE endpoint IN ('post', 'story', 'download', 'stats') AND DATE(created_at) = '$today'")->fetch_assoc()['count'];
    
    // تعداد کل دانلودها (از ابتدا تا حالا)
    $totalDownloads = $conn->query("SELECT COUNT(*) as count FROM api_calls WHERE endpoint IN ('post', 'story', 'download', 'stats')")->fetch_assoc()['count'];
    
    // تعداد کاربران جدید امروز
    $newUsersToday = $conn->query("SELECT COUNT(*) as count FROM users WHERE DATE(joined_at) = '$today'")->fetch_assoc()['count'];
    
    $conn->close();
    
    return [
        'total_users' => $totalUsers,
        'active_today' => $activeToday,
        'downloads_today' => $downloadsToday,
        'total_downloads' => $totalDownloads,
        'new_users_today' => $newUsersToday
    ];
}

/**
 * دریافت تنظیمات ربات
 * 
 * @return array تنظیمات ربات
 */
function getBotSettings() {
    $conn = getDbConnection();
    
    $settings = [];
    $result = $conn->query("SELECT setting_key, setting_value FROM settings");
    
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    
    $conn->close();
    
    return [
        'bot_active' => ($settings['bot_active'] ?? '1') == '1',
        'rate_limit' => $settings['rate_limit'] ?? RATE_LIMIT,
        'rate_limit_period' => $settings['rate_limit_period'] ?? RATE_LIMIT_PERIOD
    ];
}

/**
 * ویرایش متن پیام
 * 
 * @param int $chatId شناسه چت
 * @param int $messageId شناسه پیام
 * @param string $text متن جدید
 * @param array $keyboard کیبورد اینلاین (اختیاری)
 * @return array نتیجه درخواست
 */
function editMessageText($chatId, $messageId, $text, $keyboard = null) {
    $data = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($keyboard) {
        $data['reply_markup'] = json_encode($keyboard);
    }
    
    return sendTelegramRequest('editMessageText', $data);
}

/**
 * دریافت کیبورد پنل مدیریت
 * 
 * @return array کیبورد اینلاین پنل مدیریت
 */
function getAdminKeyboard() {
    return [
        'inline_keyboard' => [
            [
                ['text' => '📊 آمار کاربران', 'callback_data' => 'admin_stats'],
                ['text' => '⚙️ تنظیمات', 'callback_data' => 'admin_settings']
            ],
            [
                ['text' => '📣 ارسال پیام همگانی', 'callback_data' => 'admin_broadcast'],
                ['text' => '🔄 بروزرسانی ربات', 'callback_data' => 'admin_update']
            ]
        ]
    ];
}

/**
 * افزودن ادمین جدید به دیتابیس
 * 
 * @param int $telegramId شناسه تلگرام
 * @param string $username نام کاربری
 * @param string $firstName نام
 * @param string $lastName نام خانوادگی
 * @param int $addedBy شناسه ادمین اضافه‌کننده
 * @return bool نتیجه عملیات
 */
function addAdmin($telegramId, $username = '', $firstName = '', $lastName = '', $addedBy = 0) {
    $conn = getDbConnection();
    
    if (!$conn) {
        error_log("addAdmin: Database connection failed");
        return false;
    }
    
    // بررسی وجود ادمین
    $stmt = $conn->prepare("SELECT id FROM admins WHERE telegram_id = ?");
    if (!$stmt) {
        error_log("addAdmin: Prepare failed for SELECT - " . $conn->error);
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("i", $telegramId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        error_log("addAdmin: Admin already exists - telegram_id: $telegramId");
        $stmt->close();
        $conn->close();
        return false; // ادمین از قبل وجود دارد
    }
    $stmt->close();
    
    // افزودن ادمین جدید
    error_log("addAdmin: Attempting to add admin - telegram_id: $telegramId, username: $username, first_name: $firstName");
    
    $stmt = $conn->prepare("INSERT INTO admins (telegram_id, username, first_name, last_name, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
    if (!$stmt) {
        error_log("addAdmin: Prepare failed for INSERT - " . $conn->error);
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("isssi", $telegramId, $username, $firstName, $lastName, $addedBy);
    $success = $stmt->execute();
    
    if (!$success) {
        error_log("addAdmin: INSERT failed - " . $stmt->error);
        $stmt->close();
        $conn->close();
        return false;
    }
    
    error_log("addAdmin: Admin added successfully to admins table");
    $stmt->close();
    
    // به‌روزرسانی جدول users برای فلگ is_admin
    $stmt2 = $conn->prepare("UPDATE users SET is_admin = 1 WHERE telegram_id = ?");
    if ($stmt2) {
        $stmt2->bind_param("i", $telegramId);
        if ($stmt2->execute()) {
            error_log("addAdmin: Updated is_admin flag in users table");
        } else {
            error_log("addAdmin: Failed to update users table - " . $stmt2->error);
        }
        $stmt2->close();
    } else {
        error_log("addAdmin: Failed to prepare UPDATE statement - " . $conn->error);
    }
    
    $conn->close();
    error_log("addAdmin: Operation completed successfully for telegram_id: $telegramId");
    
    return true;
}

/**
 * حذف ادمین از دیتابیس
 * 
 * @param int $telegramId شناسه تلگرام
 * @return bool نتیجه عملیات
 */
function removeAdmin($telegramId) {
    // بررسی اینکه ادمین اصلی نباشه (از ADMIN_IDS)
    $mainAdminIds = explode(',', ADMIN_IDS);
    foreach ($mainAdminIds as $mainId) {
        if (trim($mainId) == $telegramId) {
            error_log("removeAdmin: Cannot remove main admin from ADMIN_IDS: " . $telegramId);
            return false; // ادمین اصلی قابل حذف نیست
        }
    }
    
    $conn = getDbConnection();
    
    if (!$conn) {
        error_log("removeAdmin: Database connection failed");
        return false;
    }
    
    error_log("removeAdmin: Attempting to remove admin with telegram_id: " . $telegramId);
    
    // حذف ادمین
    $stmt = $conn->prepare("DELETE FROM admins WHERE telegram_id = ?");
    if (!$stmt) {
        error_log("removeAdmin: Prepare failed - " . $conn->error);
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("i", $telegramId);
    $success = $stmt->execute();
    $affectedRows = $stmt->affected_rows;
    
    error_log("removeAdmin: DELETE executed - success: " . ($success ? 'true' : 'false') . ", affected_rows: " . $affectedRows);
    
    if ($success && $affectedRows > 0) {
        // به‌روزرسانی جدول users برای حذف فلگ is_admin
        $stmt2 = $conn->prepare("UPDATE users SET is_admin = 0 WHERE telegram_id = ?");
        if ($stmt2) {
            $stmt2->bind_param("i", $telegramId);
            $stmt2->execute();
            error_log("removeAdmin: Updated is_admin flag in users table");
            $stmt2->close();
        }
    } elseif ($affectedRows == 0) {
        error_log("removeAdmin: No admin found with telegram_id: " . $telegramId);
    }
    
    $stmt->close();
    $conn->close();
    
    return $success && $affectedRows > 0;
}

/**
 * دریافت لیست ادمین‌ها
 * 
 * @return array لیست ادمین‌ها
 */
function getAdminsList() {
    $conn = getDbConnection();
    
    // دریافت ادمین‌های جدول admins
    $stmt = $conn->prepare("SELECT telegram_id, username, first_name, last_name, added_at FROM admins ORDER BY added_at DESC");
    $stmt->execute();
    $result = $stmt->get_result();
    
    $admins = [];
    $adminIds = []; // برای جلوگیری از تکرار
    
    while ($row = $result->fetch_assoc()) {
        $admins[] = $row;
        $adminIds[] = $row['telegram_id'];
    }
    $stmt->close();
    
    // اضافه کردن ادمین‌های اصلی از ADMIN_IDS
    $mainAdminIds = explode(',', ADMIN_IDS);
    foreach ($mainAdminIds as $adminId) {
        $adminId = trim($adminId);
        if (empty($adminId) || in_array($adminId, $adminIds)) {
            continue; // اگر خالی یا تکراری بود رد شو
        }
        
        // دریافت اطلاعات از جدول users
        $stmt2 = $conn->prepare("SELECT telegram_id, username, first_name, last_name, joined_at as added_at FROM users WHERE telegram_id = ? LIMIT 1");
        if ($stmt2) {
            $stmt2->bind_param("i", $adminId);
            $stmt2->execute();
            $result2 = $stmt2->get_result();
            
            if ($result2->num_rows > 0) {
                $row = $result2->fetch_assoc();
                $admins[] = $row;
                $adminIds[] = $adminId;
            } else {
                // اگر توی users نبود، یه رکورد ساده بسازیم
                $admins[] = [
                    'telegram_id' => $adminId,
                    'username' => '',
                    'first_name' => 'ادمین اصلی',
                    'last_name' => '',
                    'added_at' => date('Y-m-d H:i:s')
                ];
                $adminIds[] = $adminId;
            }
            $stmt2->close();
        }
    }
    
    $conn->close();
    
    error_log("getAdminsList: Returning " . count($admins) . " admins");
    return $admins;
}

/**
 * دریافت لیست کاربران با تعداد دانلودشان
 * 
 * @return array لیست کاربران
 */
function getUsersList() {
    $conn = getDbConnection();
    
    // دریافت کاربران با تعداد دانلودها
    $query = "SELECT 
                u.id,
                u.telegram_id,
                u.username,
                u.first_name,
                u.last_name,
                u.joined_at,
                COUNT(a.id) as download_count
              FROM users u
              LEFT JOIN api_calls a ON u.telegram_id = a.telegram_id 
                AND a.endpoint IN ('post', 'story', 'download', 'stats')
              GROUP BY u.id, u.telegram_id, u.username, u.first_name, u.last_name, u.joined_at
              ORDER BY u.joined_at DESC";
    
    $result = $conn->query($query);
    
    $users = [];
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $users[] = $row;
        }
    }
    
    $conn->close();
    
    return $users;
}

/**
 * افزودن کانال جوین اجباری
 * 
 * @param string $channelUsername نام کاربری کانال
 * @param string $channelName نام نمایشی کانال
 * @param string $channelId شناسه کانال
 * @param int $addedBy شناسه ادمین اضافه‌کننده
 * @param string $inviteLink لینک دعوت کانال
 * @return bool نتیجه عملیات
 */
function addRequiredChannel($channelUsername, $channelName, $channelId = null, $addedBy = 0, $inviteLink = '') {
    $conn = getDbConnection();
    
    // حذف @ از ابتدای نام کاربری (اگر وجود داشته باشد)
    $channelUsername = ltrim($channelUsername, '@');
    
    // بررسی وجود کانال
    $stmt = $conn->prepare("SELECT id FROM channels_required WHERE channel_username = ?");
    $stmt->bind_param("s", $channelUsername);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $stmt->close();
        $conn->close();
        return false; // کانال از قبل وجود دارد
    }
    
    // افزودن کانال جدید
    $stmt = $conn->prepare("INSERT INTO channels_required (channel_username, channel_name, channel_id, invite_link, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->bind_param("ssssi", $channelUsername, $channelName, $channelId, $inviteLink, $addedBy);
    $success = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $success;
}

/**
 * حذف کانال جوین اجباری
 * 
 * @param string $channelUsername نام کاربری کانال
 * @return bool نتیجه عملیات
 */
function removeRequiredChannel($channelUsername) {
    $conn = getDbConnection();
    
    // حذف @ از ابتدای نام کاربری (اگر وجود داشته باشد)
    $channelUsername = ltrim($channelUsername, '@');
    
    // حذف کانال
    $stmt = $conn->prepare("DELETE FROM channels_required WHERE channel_username = ?");
    $stmt->bind_param("s", $channelUsername);
    $success = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $success;
}

/**
 * دریافت لیست کانال‌های جوین اجباری
 * 
 * @return array لیست کانال‌ها
 */
function getRequiredChannelsList() {
    $conn = getDbConnection();
    
    $stmt = $conn->prepare("SELECT channel_username, channel_name, channel_id, invite_link, added_at FROM channels_required ORDER BY added_at DESC");
    $stmt->execute();
    $result = $stmt->get_result();
    
    $channels = [];
    while ($row = $result->fetch_assoc()) {
        $channels[] = $row;
    }
    
    $stmt->close();
    $conn->close();
    
    return $channels;
}

/**
 * ارسال پیام یا فوروارد همگانی به کاربران
 * 
 * @param string $messageText متن پیام (برای ارسال همگانی)
 * @param int $fromChatId شناسه چت مبدا (برای فوروارد)
 * @param int $messageId شناسه پیام (برای فوروارد)
 * @param int $sentBy شناسه ادمین ارسال‌کننده
 * @param bool $isForward آیا فوروارد است
 * @return array آمار ارسال
 */
function sendBroadcast($messageText = '', $fromChatId = null, $messageId = null, $sentBy = 0, $isForward = false) {
    $conn = getDbConnection();
    
    // دریافت لیست کاربران
    $result = $conn->query("SELECT telegram_id FROM users WHERE telegram_id != 0");
    
    $totalUsers = $result->num_rows;
    $successfulSends = 0;
    $failedSends = 0;
    
    // ثبت برودکست در دیتابیس
    $stmt = $conn->prepare("INSERT INTO broadcasts (message, sent_by, total_users, is_forward, message_id, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->bind_param("siiii", $messageText, $sentBy, $totalUsers, $isForward, $messageId);
    $stmt->execute();
    $broadcastId = $conn->insert_id;
    $stmt->close();
    
    // ارسال پیام به کاربران
    while ($row = $result->fetch_assoc()) {
        $chatId = $row['telegram_id'];
        
        if ($isForward && $fromChatId && $messageId) {
            // فوروارد پیام
            $sendResult = forwardMessage($chatId, $fromChatId, $messageId);
        } else {
            // ارسال پیام معمولی
            $sendResult = sendMessage($chatId, $messageText);
        }
        
        if ($sendResult['success']) {
            $successfulSends++;
        } else {
            $failedSends++;
        }
        
        // تاخیر کوچک برای جلوگیری از محدودیت API
        usleep(50000); // 50ms
    }
    
    // به‌روزرسانی آمار برودکست
    $stmt = $conn->prepare("UPDATE broadcasts SET successful_sends = ?, failed_sends = ?, completed_at = NOW() WHERE id = ?");
    $stmt->bind_param("iii", $successfulSends, $failedSends, $broadcastId);
    $stmt->execute();
    $stmt->close();
    
    $conn->close();
    
    return [
        'total' => $totalUsers,
        'successful' => $successfulSends,
        'failed' => $failedSends
    ];
}

/**
 * دریافت اطلاعات کانال از تلگرام
 * 
 * @param string $channelId شناسه یا نام کاربری کانال
 * @return array نتیجه درخواست
 */
function getChat($channelId) {
    $data = [
        'chat_id' => $channelId
    ];
    
    return sendTelegramRequest('getChat', $data);
}

/**
 * دریافت اطلاعات پروفایل اینستاگرام
 * 
 * @param string $username نام کاربری
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 * @return bool نتیجه عملیات
 */
function getInstagramProfile($username, $chatId, $userId) {
    // ارسال پیام در حال پردازش
    sendMessage($chatId, "⏳ در حال دریافت اطلاعات پروفایل <b>@$username</b>...");
    
    // فراخوانی API اینستاگرام - تلاش با type های مختلف
    // ابتدا با info امتحان می‌کنیم
    $response = callInstagramAPI('info', ['username' => $username]);
    
    // اگر موفق نبود، با user امتحان می‌کنیم
    if (!$response['success'] || (isset($response['data']['ok']) && !$response['data']['ok'])) {
        $response = callInstagramAPI('profile', ['username' => $username]);
    }
    
    if (!$response['success']) {
        sendMessage($chatId, "❌ خطا در دریافت اطلاعات پروفایل. لطفاً دوباره تلاش کنید.");
        return false;
    }
    
    $data = $response['data'];
    
    // بررسی وجود داده - پشتیبانی از فرمت‌های مختلف
    $user = null;
    
    if (isset($data['status']) && $data['status'] === 'ok' && isset($data['result'])) {
        $user = $data['result'];
    } elseif (isset($data['result']) && $data['result'] === 'ok' && isset($data['data'])) {
        $user = $data['data'];
    } elseif (isset($data['user'])) {
        $user = $data['user'];
    } elseif (isset($data['username'])) {
        $user = $data;
    }
    
    if (!$user) {
        sendMessage($chatId, "❌ پروفایل یافت نشد یا خصوصی است.");
        return false;
    }
    
    $profileMessage = "👤 <b>اطلاعات پروفایل</b>\n\n";
    $profileMessage .= "🔹 نام کاربری: @" . $username . "\n";
    
    if (isset($user['full_name']) && !empty($user['full_name'])) {
        $profileMessage .= "🔹 نام: " . $user['full_name'] . "\n";
    }
    
    if (isset($user['biography']) && !empty($user['biography'])) {
        $profileMessage .= "🔹 بیو: " . $user['biography'] . "\n";
    }
    
    $profileMessage .= "\n📊 <b>آمار:</b>\n";
    
    // پشتیبانی از فرمت‌های مختلف آمار
    $followerCount = 0;
    $followingCount = 0;
    $mediaCount = 0;
    
    if (isset($user['follower_count'])) {
        $followerCount = $user['follower_count'];
    } elseif (isset($user['edge_followed_by']['count'])) {
        $followerCount = $user['edge_followed_by']['count'];
    }
    
    if (isset($user['following_count'])) {
        $followingCount = $user['following_count'];
    } elseif (isset($user['edge_follow']['count'])) {
        $followingCount = $user['edge_follow']['count'];
    }
    
    if (isset($user['media_count'])) {
        $mediaCount = $user['media_count'];
    } elseif (isset($user['edge_owner_to_timeline_media']['count'])) {
        $mediaCount = $user['edge_owner_to_timeline_media']['count'];
    }
    
    if ($followerCount > 0) {
        $profileMessage .= "👥 فالوورها: " . number_format($followerCount) . "\n";
    }
    
    if ($followingCount > 0) {
        $profileMessage .= "👣 فالووینگ: " . number_format($followingCount) . "\n";
    }
    
    if ($mediaCount > 0) {
        $profileMessage .= "📷 پست‌ها: " . number_format($mediaCount) . "\n";
    }
    
    if (isset($user['is_private']) && $user['is_private']) {
        $profileMessage .= "\n🔒 این اکانت <b>خصوصی</b> است.";
    }
    
    if (isset($user['is_verified']) && $user['is_verified']) {
        $profileMessage .= "\n✅ این اکانت <b>تأیید شده</b> است.";
    }
    
    // ارسال عکس پروفایل اگر موجود است
    if (isset($user['profile_pic_url'])) {
        sendPhoto($chatId, $user['profile_pic_url'], $profileMessage);
    } else {
        sendMessage($chatId, $profileMessage);
    }
    
    return true;
}

/**
 * دانلود و ارسال مدیا
 * 
 * @param int $chatId شناسه چت
 * @param int $userId شناسه کاربر
 * @param string $type نوع مدیا
 * @param string $url آدرس مدیا
 * @return bool نتیجه عملیات
 */
function downloadAndSendMedia($chatId, $userId, $type, $url) {
    // ارسال پیام در حال دانلود
    sendMessage($chatId, "⏳ در حال دانلود...");
    
    // بررسی نوع مدیا
    if ($type === 'video' || strpos($url, '.mp4') !== false) {
        sendVideo($chatId, $url, "🎬 ویدیو درخواستی شما");
    } elseif ($type === 'photo' || $type === 'image') {
        sendPhoto($chatId, $url, "🖼 تصویر درخواستی شما");
    } else {
        // تلاش برای ارسال به عنوان ویدیو
        $result = sendVideo($chatId, $url, "📥 مدیای درخواستی شما");
        
        // اگر ارسال ویدیو موفق نبود، به عنوان عکس ارسال کن
        if (!$result['success']) {
            sendPhoto($chatId, $url, "📥 مدیای درخواستی شما");
        }
    }
    
    return true;
}

/**
 * دانلود و آپلود ویدیو (برای URL های خیلی طولانی)
 * 
 * @param int $chatId شناسه چت
 * @param string $videoUrl آدرس ویدیو
 * @param string $caption توضیحات
 * @return bool نتیجه عملیات
 */
function downloadAndUploadVideo($chatId, $videoUrl, $caption = '') {
    try {
        // ارسال پیام "در حال دانلود"
        sendMessage($chatId, "⏳ در حال دانلود ویدیو...");
        
        $maxRetries = 3; // تعداد تلاش‌ها
        $retryDelay = 2; // تأخیر بین تلاش‌ها (ثانیه)
        $videoContent = false;
        $httpCode = 0;
        $error = '';
        
        // تلاش برای دانلود با retry
        for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
            if ($attempt > 1) {
                error_log("Download attempt $attempt of $maxRetries...");
                sleep($retryDelay * $attempt); // exponential backoff
            }
            
            // دانلود ویدیو با cURL
            $ch = curl_init($videoUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 90); // افزایش به 90 ثانیه
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); // timeout برای اتصال
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Accept: video/mp4,video/*,*/*',
                'Accept-Language: en-US,en;q=0.9',
                'Referer: https://www.instagram.com/',
                'Cache-Control: no-cache'
            ]);
            
            $videoContent = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            // اگر موفق شد، از loop خارج شو
            if ($videoContent !== false && !empty($videoContent) && $httpCode === 200) {
                error_log("Download successful on attempt $attempt");
                break;
            }
            
            error_log("Download failed on attempt $attempt. HTTP Code: $httpCode, Error: $error");
        }
        
        if ($videoContent === false || empty($videoContent) || $httpCode !== 200) {
            error_log("Failed to download video after $maxRetries attempts. HTTP Code: $httpCode, Error: $error");
            
            // اگر دانلود موفق نبود، لینک بده
            $errorMsg = $caption . "\n\n";
            $errorMsg .= "⚠️ خطا در دانلود خودکار ویدیو (پس از $maxRetries تلاش).\n\n";
            $errorMsg .= "🔗 <b>لینک دانلود مستقیم:</b>";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '⬇️ دانلود مستقیم', 'url' => $videoUrl]]
                ]
            ];
            
            sendMessage($chatId, $errorMsg, $keyboard);
            return false;
        }
        
        // ذخیره موقت
        $tempFile = sys_get_temp_dir() . '/story_' . time() . '_' . rand(1000, 9999) . '.mp4';
        file_put_contents($tempFile, $videoContent);
        
        $fileSize = filesize($tempFile);
        error_log("Video downloaded to: " . $tempFile . " (size: " . $fileSize . " bytes)");
        
        // بررسی سایز فایل
        if ($fileSize === 0 || $fileSize > 50 * 1024 * 1024) { // بیشتر از 50MB
            @unlink($tempFile);
            
            if ($fileSize === 0) {
                error_log("Downloaded file is empty!");
            } else {
                error_log("File too large: $fileSize bytes");
            }
            
            // لینک بده
            $errorMsg = $caption . "\n\n";
            $errorMsg .= ($fileSize === 0 ? "⚠️ فایل خالی است.\n\n" : "⚠️ فایل خیلی بزرگ است (بیش از 50MB).\n\n");
            $errorMsg .= "🔗 <b>لینک دانلود مستقیم:</b>";
            
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '⬇️ دانلود مستقیم', 'url' => $videoUrl]]
                ]
            ];
            
            sendMessage($chatId, $errorMsg, $keyboard);
            return false;
        }
        
        // آپلود به تلگرام با retry
        $uploadResult = uploadVideoToTelegramWithRetry($chatId, $tempFile, $caption);
        
        // حذف فایل موقت
        @unlink($tempFile);
        
        return $uploadResult;
        
    } catch (Exception $e) {
        error_log("Exception in downloadAndUploadVideo: " . $e->getMessage());
        
        // اگر خطا داد، لینک بده
        $errorMsg = $caption . "\n\n";
        $errorMsg .= "⚠️ خطا در پردازش.\n\n";
        $errorMsg .= "🔗 <b>لینک دانلود مستقیم:</b>";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '⬇️ دانلود مستقیم', 'url' => $videoUrl]]
            ]
        ];
        
        sendMessage($chatId, $errorMsg, $keyboard);
        return false;
    }
}

/**
 * آپلود ویدیو به تلگرام با CURLFile
 * 
 * @param int $chatId شناسه چت
 * @param string $filePath مسیر فایل
 * @param string $caption توضیحات
 * @return bool نتیجه عملیات
 */
function uploadVideoToTelegram($chatId, $filePath, $caption = '') {
    $url = TELEGRAM_API_URL . '/sendVideo';
    
    $postData = [
        'chat_id' => $chatId,
        'video' => new CURLFile($filePath, 'video/mp4', 'story.mp4'),
        'caption' => $caption,
        'parse_mode' => 'HTML'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120); // 2 دقیقه برای آپلود
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($error) {
        error_log("Upload error: $error");
        return false;
    }
    
    $result = json_decode($response, true);
    
    if (!$result || !$result['ok']) {
        $errorMsg = isset($result['description']) ? $result['description'] : 'Unknown error';
        error_log("Telegram upload error: $errorMsg");
        return false;
    }
    
    error_log("Video uploaded successfully!");
    return true;
}

/**
 * آپلود ویدیو به تلگرام با retry logic
 * 
 * @param int $chatId شناسه چت
 * @param string $filePath مسیر فایل ویدیو
 * @param string $caption کپشن
 * @return bool نتیجه آپلود
 */
function uploadVideoToTelegramWithRetry($chatId, $filePath, $caption = '') {
    $maxRetries = 2; // 2 تلاش اضافی
    
    for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
        if ($attempt > 1) {
            error_log("Upload attempt $attempt of $maxRetries...");
            sleep(3); // 3 ثانیه صبر
        }
        
        $result = uploadVideoToTelegram($chatId, $filePath, $caption);
        
        if ($result) {
            return true;
        }
        
        error_log("Upload failed on attempt $attempt");
    }
    
    error_log("Upload failed after $maxRetries attempts");
    return false;
}

/**
 * پاسخ به کالبک‌کوئری
 * 
 * @param string $callbackId شناسه کالبک
 * @param string $text متن پاسخ
 * @param bool $showAlert نمایش به صورت هشدار
 * @return array نتیجه درخواست
 */
function answerCallbackQuery($callbackId, $text = '', $showAlert = false) {
    $data = [
        'callback_query_id' => $callbackId,
        'text' => $text,
        'show_alert' => $showAlert
    ];
    
    return sendTelegramRequest('answerCallbackQuery', $data);
}