<?php
/**
 * Group Manager Handler
 * Version: 2.0
 */

declare(strict_types=1);

class GroupManager {
    private Bot $bot;
    private Telegram $telegram;
    private Database $db;
    private ?array $groupData = null;
    
    public function __construct(Bot $bot) {
        $this->bot = $bot;
        $this->telegram = $bot->getTelegram();
        $this->db = $bot->getDb();
        $this->loadGroupData();
    }
    
    private function loadGroupData(): void {
        $this->groupData = $this->db->fetchOne(
            "SELECT * FROM `groups` WHERE id = ?",
            [$this->bot->getChatId()]
        );
    }
    
    public function handle(): void {
        $text = $this->bot->getText();
        
        if (!$text) {
            $this->checkLocks();
            return;
        }
        
        $command = $this->parseCommand($text);
        
        // Install command
        if ($command === 'start' || $command === 'نصب') {
            $this->installBot();
            return;
        }
        
        // Check if bot is installed
        if (!$this->groupData) {
            return;
        }
        
        // Check locks before processing commands
        if (!$this->isManager()) {
            $this->checkLocks();
        }
        
        // Process commands for managers
        if ($this->isManager()) {
            $this->handleManagerCommands($command);
        }
    }
    
    private function installBot(): void {
        if ($this->groupData) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                "✅ ربات قبلاً در این گروه نصب شده است!"
            );
            return;
        }
        
        // Get admins
        $admins = $this->telegram->getChatAdministrators($this->bot->getChatId());
        $adminIds = [];
        $creatorId = null;
        
        if ($admins) {
            foreach ($admins as $admin) {
                $adminIds[] = $admin->user->id;
                if ($admin->status === 'creator') {
                    $creatorId = $admin->user->id;
                }
            }
        }
        
        // Insert group
        $this->db->insert('groups', [
            'id' => $this->bot->getChatId(),
            'creator' => $creatorId,
            'installer' => $this->bot->getUserId(),
            'promote' => json_encode($adminIds),
            'warn' => json_encode(['warn' => 3, 'type' => 'ban']),
            'flood' => json_encode(['max' => 5, 'time' => 5, 'type' => 'silent']),
            'locked' => json_encode([
                'link' => 'on',
                'badword' => 'on',
                'bot' => 'on',
                'tag' => 'on',
                'addbot' => 'on'
            ])
        ]);
        
        $message = "✅ *ربات با موفقیت نصب شد!*\n\n" .
                   "👤 مالک: [{$creatorId}](tg://user?id={$creatorId})\n" .
                   "👥 تعداد مدیران: " . count($adminIds) . "\n\n" .
                   "📝 برای مشاهده دستورات: `/help`\n" .
                   "⚙️ برای تنظیمات: `/panel`\n\n" .
                   "📢 کانال: " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message
        );
    }
    
    private function handleManagerCommands(string $command): void {
        switch ($command) {
            case 'help':
            case 'راهنما':
                $this->showHelp();
                break;
                
            case 'panel':
            case 'پنل':
                $this->showPanel();
                break;
                
            case 'settings':
            case 'تنظیمات':
                $this->showSettings();
                break;
                
            case 'lock':
            case 'قفل':
                $this->handleLock();
                break;
                
            case 'unlock':
            case 'بازکردن':
                $this->handleUnlock();
                break;
        }
    }
    
    private function showHelp(): void {
        $message = "📚 *راهنمای دستورات*\n\n" .
                   "*مدیریت کاربران:*\n" .
                   "• `/promote` - ترفیع مدیر\n" .
                   "• `/demote` - عزل مدیر\n" .
                   "• `/ban` - بن کاربر\n" .
                   "• `/unban` - آنبن کاربر\n" .
                   "• `/silent` - سکوت کاربر\n" .
                   "• `/unsilent` - حذف سکوت\n" .
                   "• `/warn` - اخطار\n" .
                   "• `/kick` - اخراج\n\n" .
                   "*قفل‌ها:*\n" .
                   "• `/lock [type]` - قفل کردن\n" .
                   "• `/unlock [type]` - باز کردن\n\n" .
                   "*تنظیمات:*\n" .
                   "• `/panel` - پنل مدیریتی\n" .
                   "• `/settings` - تنظیمات\n\n" .
                   "📢 کانال: " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message
        );
    }
    
    private function showPanel(): void {
        $locks = json_decode($this->groupData['locked'] ?? '{}', true);
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => ($locks['link'] ?? '') === 'on' ? '🔒 لینک' : '🔓 لینک', 'callback_data' => 'toggle_link'],
                    ['text' => ($locks['badword'] ?? '') === 'on' ? '🔒 فحش' : '🔓 فحش', 'callback_data' => 'toggle_badword']
                ],
                [
                    ['text' => ($locks['bot'] ?? '') === 'on' ? '🔒 ربات' : '🔓 ربات', 'callback_data' => 'toggle_bot'],
                    ['text' => ($locks['tag'] ?? '') === 'on' ? '🔒 تگ' : '🔓 تگ', 'callback_data' => 'toggle_tag']
                ],
                [
                    ['text' => '👥 لیست‌ها', 'callback_data' => 'lists'],
                    ['text' => '📊 آمار', 'callback_data' => 'stats']
                ],
                [
                    ['text' => '❌ بستن', 'callback_data' => 'close']
                ]
            ]
        ];
        
        $message = "🎛 *پنل مدیریتی گروه*\n\n";
        $message .= "🔒 = قفل فعال\n";
        $message .= "🔓 = قفل غیرفعال\n\n";
        $message .= "روی هر دکمه کلیک کنید تا وضعیت آن تغییر کند.";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showSettings(): void {
        $locks = json_decode($this->groupData['locked'] ?? '{}', true);
        
        $message = "⚙️ *تنظیمات گروه*\n\n";
        $message .= "*قفل‌های فعال:*\n";
        
        foreach ($locks as $lock => $status) {
            if ($status === 'on') {
                $message .= "✅ {$lock}\n";
            }
        }
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message
        );
    }
    
    private function checkLocks(): void {
        // Implementation for checking locks
        // This will check all active locks and delete message if needed
        // TODO: Implement lock checking
    }
    
    private function handleLock(): void {
        // Implementation for lock command
    }
    
    private function handleUnlock(): void {
        // Implementation for unlock command
    }
    
    public function handleCallback(): void {
        $data = $this->bot->getCallbackData();
        
        // Handle toggle locks
        if (preg_match('/^toggle_(.+)$/', $data, $matches)) {
            $this->toggleLock($matches[1]);
            return;
        }
        
        // Answer callback
        $this->telegram->answerCallbackQuery($this->bot->getCallbackQuery()->id);
        
        switch ($data) {
            case 'close':
                $this->telegram->deleteMessage(
                    $this->bot->getChatId(),
                    $this->bot->getMessageId()
                );
                break;
                
            case 'locks':
                $this->showLocksPanel();
                break;
                
            case 'lists':
                $this->showListsPanel();
                break;
                
            case 'settings':
                $this->showSettings();
                break;
                
            case 'stats':
                $this->showGroupStats();
                break;
                
            default:
                $this->telegram->answerCallbackQuery(
                    $this->bot->getCallbackQuery()->id,
                    "این قابلیت در حال توسعه است"
                );
                break;
        }
    }
    
    /**
     * Toggle lock on/off
     */
    private function toggleLock(string $lockType): void {
        $locks = json_decode($this->groupData['locked'] ?? '{}', true);
        
        // Toggle the lock
        $currentStatus = $locks[$lockType] ?? 'off';
        $locks[$lockType] = $currentStatus === 'on' ? 'off' : 'on';
        
        // Update database
        $this->db->update('groups', [
            'locked' => json_encode($locks)
        ], 'id = ?', [$this->bot->getChatId()]);
        
        // Reload group data
        $this->loadGroupData();
        
        // Answer callback
        $lockNames = [
            'link' => 'لینک',
            'badword' => 'فحش',
            'bot' => 'ربات',
            'tag' => 'تگ',
            'addbot' => 'اضافه کردن ربات'
        ];
        
        $name = $lockNames[$lockType] ?? $lockType;
        $status = $locks[$lockType] === 'on' ? 'فعال' : 'غیرفعال';
        
        $this->telegram->answerCallbackQuery(
            $this->bot->getCallbackQuery()->id,
            "قفل {$name}: {$status}"
        );
        
        // Update panel
        $this->updatePanel();
    }
    
    /**
     * Update panel with new lock status
     */
    private function updatePanel(): void {
        $locks = json_decode($this->groupData['locked'] ?? '{}', true);
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => ($locks['link'] ?? '') === 'on' ? '🔒 لینک' : '🔓 لینک', 'callback_data' => 'toggle_link'],
                    ['text' => ($locks['badword'] ?? '') === 'on' ? '🔒 فحش' : '🔓 فحش', 'callback_data' => 'toggle_badword']
                ],
                [
                    ['text' => ($locks['bot'] ?? '') === 'on' ? '🔒 ربات' : '🔓 ربات', 'callback_data' => 'toggle_bot'],
                    ['text' => ($locks['tag'] ?? '') === 'on' ? '🔒 تگ' : '🔓 تگ', 'callback_data' => 'toggle_tag']
                ],
                [
                    ['text' => '👥 لیست‌ها', 'callback_data' => 'lists'],
                    ['text' => '📊 آمار', 'callback_data' => 'stats']
                ],
                [
                    ['text' => '❌ بستن', 'callback_data' => 'close']
                ]
            ]
        ];
        
        $message = "🎛 *پنل مدیریتی گروه*\n\n";
        $message .= "🔒 = قفل فعال\n";
        $message .= "🔓 = قفل غیرفعال\n\n";
        $message .= "روی هر دکمه کلیک کنید تا وضعیت آن تغییر کند.";
        
        $this->telegram->editMessageText(
            $this->bot->getChatId(),
            $this->bot->getMessageId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showLocksPanel(): void {
        $locks = json_decode($this->groupData['locked'] ?? '{}', true);
        
        $message = "🔒 *قفل‌های گروه*\n\n";
        
        $lockNames = [
            'link' => 'لینک',
            'badword' => 'فحش',
            'bot' => 'ربات',
            'tag' => 'تگ',
            'addbot' => 'اضافه کردن ربات'
        ];
        
        foreach ($lockNames as $key => $name) {
            $status = isset($locks[$key]) && $locks[$key] === 'on' ? '🔒' : '🔓';
            $message .= "{$status} {$name}\n";
        }
        
        $message .= "\n📝 برای تغییر از دستورات زیر استفاده کنید:\n";
        $message .= "`/lock [نوع]` - قفل کردن\n";
        $message .= "`/unlock [نوع]` - باز کردن";
        
        $this->telegram->editMessageText(
            $this->bot->getChatId(),
            $this->bot->getMessageId(),
            $message,
            'Markdown'
        );
    }
    
    private function showListsPanel(): void {
        $promote = json_decode($this->groupData['promote'] ?? '[]', true);
        $ban = json_decode($this->groupData['ban'] ?? '[]', true);
        $silent = json_decode($this->groupData['silent'] ?? '[]', true);
        
        $message = "📋 *لیست‌های گروه*\n\n";
        $message .= "👥 مدیران: " . count($promote) . " نفر\n";
        $message .= "🚫 بن شده: " . count($ban) . " نفر\n";
        $message .= "🔇 سکوت: " . count($silent) . " نفر\n";
        
        $this->telegram->editMessageText(
            $this->bot->getChatId(),
            $this->bot->getMessageId(),
            $message,
            'Markdown'
        );
    }
    
    private function showGroupStats(): void {
        $message = "📊 *آمار گروه*\n\n";
        $message .= "🆔 شناسه: `{$this->bot->getChatId()}`\n";
        $message .= "📅 نصب شده: " . ($this->groupData['created_at'] ?? 'نامشخص') . "\n";
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown'
        );
    }
    
    private function parseCommand(string $text): string {
        $text = trim($text);
        $text = preg_replace('/^[\/\!\?#@]+/', '', $text);
        $text = preg_replace('/@' . BOT_USERNAME . '$/i', '', $text);
        return strtolower($text);
    }
    
    private function isManager(): bool {
        if ($this->bot->isAdmin()) {
            return true;
        }
        
        $promote = json_decode($this->groupData['promote'] ?? '[]', true);
        return in_array($this->bot->getUserId(), $promote);
    }
}
