<?php
/**
 * Private Chat Handler
 * Version: 2.0
 */

declare(strict_types=1);

class PrivateChat {
    private Bot $bot;
    private Telegram $telegram;
    private Database $db;
    
    public function __construct(Bot $bot) {
        $this->bot = $bot;
        $this->telegram = $bot->getTelegram();
        $this->db = $bot->getDb();
    }
    
    public function handle(): void {
        $text = $this->bot->getText();
        $chatId = $this->bot->getChatId();
        
        if (!$text) {
            return;
        }
        
        $command = $this->parseCommand($text);
        
        switch ($command) {
            case '/start':
            case 'start':
                $this->showStart();
                break;
                
            case 'help':
            case 'راهنما':
            case 'راهنما':
                $this->showHelp();
                break;
                
            case 'support':
            case 'پشتیبانی':
            case 'پشتیبانی':
                $this->showSupport();
                break;
                
            case 'stats':
            case 'آمار':
            case 'امار':
                $this->showStats();
                break;
                
            case 'channel':
            case 'کانال':
                $this->telegram->sendMessage(
                    $this->bot->getChatId(),
                    "📢 کانال ما: " . SUPPORT_CHANNEL
                );
                break;
                
            case 'language':
            case 'زبان':
                $this->showLanguageMenu();
                break;
                
            case 'ai':
            case 'هوش_مصنوعی':
                $this->showAIMenu();
                break;
                
            case 'plugins':
            case 'پلاگین':
                if ($this->bot->isAdmin()) {
                    $this->showPluginsMenu();
                }
                break;
                
            default:
                // Try AI response if enabled
                if (AI_ENABLED) {
                    $this->handleAIChat($text);
                } else {
                    $this->showStart();
                }
        }
    }
    
    private function parseCommand(string $text): string {
        $text = trim($text);
        // Remove emojis
        $text = preg_replace('/[\x{1F300}-\x{1F9FF}]/u', '', $text);
        $text = preg_replace('/^[\/\!\?#@]+/', '', $text);
        $text = preg_replace('/@' . BOT_USERNAME . '$/i', '', $text);
        $text = trim($text);
        return strtolower($text);
    }
    
    private function showStart(): void {
        // Check subscription status
        require_once 'AccessControl.php';
        $access = AccessControl::getInstance();
        $status = $access->getAccessStatus($this->bot->getUserId());
        
        $keyboard = [
            'keyboard' => [
                [['text' => '📊 آمار'], ['text' => '📚 راهنما']],
                [['text' => '💬 پشتیبانی'], ['text' => '📢 کانال']],
            ],
            'resize_keyboard' => true
        ];
        
        // Add subscription button if needed
        if (SUBSCRIPTION_REQUIRED && !$status['is_admin']) {
            $keyboard['keyboard'][] = [['text' => '💎 اشتراک'], ['text' => '📊 اشتراک من']];
        }
        
        $message = "🤖 *به ربات مدیریت گروه خوش آمدید!*\n\n" .
                   "برای استفاده از ربات، آن را به گروه خود اضافه کنید:\n" .
                   "👉 [افزودن به گروه](https://t.me/" . BOT_USERNAME . "?startgroup=new)\n\n" .
                   "✅ ربات را ادمین کنید\n" .
                   "✅ دستور /start را در گروه ارسال کنید\n\n";
        
        // Add subscription info
        if (SUBSCRIPTION_REQUIRED && !$status['is_admin']) {
            if ($status['in_trial']) {
                $message .= "🎁 *دوره آزمایشی رایگان*\n" .
                           "⏳ {$status['trial_days_left']} روز باقیمانده\n\n";
            } elseif ($status['has_subscription']) {
                $endDate = date('Y/m/d', strtotime($status['subscription_end']));
                $message .= "✅ *اشتراک فعال*\n" .
                           "📅 تا: {$endDate}\n\n";
            } else {
                $message .= "⚠️ *نیاز به اشتراک*\n" .
                           "برای استفاده کامل: /subscribe\n\n";
            }
        }
        
        $message .= "📢 کانال: " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showHelp(): void {
        $message = "📚 *راهنمای استفاده از ربات*\n\n" .
                   "*دستورات اصلی:*\n" .
                   "• `/start` - نصب ربات در گروه\n" .
                   "• `/help` - راهنمای دستورات\n" .
                   "• `/panel` - پنل مدیریتی\n" .
                   "• `/settings` - تنظیمات گروه\n\n" .
                   "*قابلیت‌های ربات:*\n" .
                   "✅ مدیریت کاربران (بن، سکوت، اخطار)\n" .
                   "✅ قفل انواع محتوا\n" .
                   "✅ ضد رگبار\n" .
                   "✅ عضویت اجباری کانال\n" .
                   "✅ خوشامدگویی\n" .
                   "✅ فیلتر کلمات\n" .
                   "✅ و بیشتر...\n\n" .
                   "📢 کانال: " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message
        );
    }
    
    private function showSupport(): void {
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '💬 گروه پشتیبانی', 'url' => SUPPORT_GROUP]],
                [['text' => '📢 کانال', 'url' => 'https://t.me/' . ltrim(SUPPORT_CHANNEL, '@')]]
            ]
        ];
        
        $message = "💬 *پشتیبانی*\n\n" .
                   "برای دریافت پشتیبانی و گزارش مشکلات:\n\n" .
                   "👥 گروه پشتیبانی: " . SUPPORT_GROUP . "\n" .
                   "📢 کانال: " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message,
            'Markdown',
            $keyboard
        );
    }
    
    private function showStats(): void {
        $totalGroups = $this->db->query("SELECT COUNT(*) as count FROM `groups`")->fetch_assoc()['count'];
        $totalMembers = $this->db->query("SELECT COUNT(*) as count FROM `members`")->fetch_assoc()['count'];
        
        $message = __('stats.title') . "\n\n" .
                   "👥 " . __('stats.total_users') . ": `{$totalMembers}`\n" .
                   "🏘 " . __('stats.total_groups') . ": `{$totalGroups}`\n\n" .
                   "📢 " . __('welcome.support') . ": " . SUPPORT_CHANNEL;
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            $message
        );
    }
    
    private function showLanguageMenu(): void {
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🇮🇷 فارسی', 'callback_data' => 'lang_fa'],
                    ['text' => '🇬🇧 English', 'callback_data' => 'lang_en']
                ],
                [
                    ['text' => '🇸🇦 العربية', 'callback_data' => 'lang_ar'],
                    ['text' => '🇹🇷 Türkçe', 'callback_data' => 'lang_tr']
                ]
            ]
        ];
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            __('language.select'),
            'Markdown',
            $keyboard
        );
    }
    
    private function showAIMenu(): void {
        // Check access
        require_once 'AccessControl.php';
        $access = AccessControl::getInstance();
        
        if (!$access->requireSubscription($this->bot->getUserId(), $this->bot->getChatId(), 'ai_chat')) {
            return;
        }
        
        if (!AI_ENABLED) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                __('ai.no_api_key')
            );
            return;
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '💬 چت با AI', 'callback_data' => 'ai_chat'],
                    ['text' => '🔍 تحلیل احساسات', 'callback_data' => 'ai_sentiment']
                ],
                [
                    ['text' => '📝 خلاصه‌سازی', 'callback_data' => 'ai_summarize'],
                    ['text' => '🌐 ترجمه', 'callback_data' => 'ai_translate']
                ],
                [
                    ['text' => '🗑 پاک کردن تاریخچه', 'callback_data' => 'ai_clear']
                ]
            ]
        ];
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            "🤖 *منوی هوش مصنوعی*\n\nلطفاً یکی از گزینه‌ها را انتخاب کنید:",
            'Markdown',
            $keyboard
        );
    }
    
    private function showPluginsMenu(): void {
        $pluginManager = PluginManager::getInstance();
        $plugins = $pluginManager->getAllPlugins();
        
        if (empty($plugins)) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                __('plugins.no_plugins')
            );
            return;
        }
        
        $keyboard = [];
        foreach ($plugins as $name => $plugin) {
            $enabled = $pluginManager->isEnabled($name);
            $status = $enabled ? '✅' : '❌';
            $keyboard[] = [
                ['text' => "{$status} {$name}", 'callback_data' => "plugin_toggle_{$name}"]
            ];
        }
        
        $this->telegram->sendMessage(
            $this->bot->getChatId(),
            __('plugins.list'),
            'Markdown',
            ['inline_keyboard' => $keyboard]
        );
    }
    
    private function handleAIChat(string $text): void {
        // Check access
        require_once 'AccessControl.php';
        $access = AccessControl::getInstance();
        
        if (!$access->requireSubscription($this->bot->getUserId(), $this->bot->getChatId(), 'ai_chat')) {
            return;
        }
        
        $ai = GeminiAI::getInstance();
        
        // Send thinking message
        $thinking = $this->telegram->sendMessage(
            $this->bot->getChatId(),
            __('ai.thinking')
        );
        
        // Get AI response
        $response = $ai->chat($text, $this->bot->getUserId());
        
        // Delete thinking message
        if ($thinking && isset($thinking->result->message_id)) {
            $this->telegram->deleteMessage(
                $this->bot->getChatId(),
                $thinking->result->message_id
            );
        }
        
        // Send response
        if ($response) {
            $this->telegram->sendMessage(
                $this->bot->getChatId(),
                $response
            );
        }
    }
}
